/* ****************************************************************************
  This file is part of KBabel

  Copyright (C) 1999 by Matthias Kiefer
                            <matthias.kiefer@gmx.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

**************************************************************************** */
#ifndef SEARCHRESULTBOX_H
#define SEARCHRESULTBOX_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qwidget.h>
#include <qlist.h>
#include <qvaluelist.h>
#include <qfile.h>
#include "catalogitem.h"
#include "settings.h"

class QMultiLineEdit;
class QPushButton;
class QLabel;
class KLocale;
class QString;
class QPopupMenu;

struct SearchResultItem
{
   // The found item
   CatalogItem item;

   // The package it was found in
   QString  package;
};




class SearchResultBox : public QWidget
{
   Q_OBJECT
public:
   SearchResultBox(QWidget* parent=0,const char* name=0);
   virtual ~SearchResultBox();

   /** starts search with mode mode*/
   void search(QString msgid,SearchMode mode);

   /**
   * this method starts a timer and will start the search after
   * 5 ms, This is a little workaround, when opened a new file
   * to let other windows update their contents before starting the search
   */
   void startDelayedSearch(QString msgid,SearchMode mode);

   bool isSearching(){return _searching;}

   /** reads preferences from the apps config object */
   void readSettings();
   /** writes preferences to the apps config object */
   void saveSettings();

   SearchSettings settings() const{return _settings;}

   /**
   * this should be called when a new file is opened or the language changed
   * in the editor to check the given names of the search files.
   * If necessary new files are opened
   * @param packageName The name of the opened catalog(without extension)
   */
   void setPackageName(const QString& packageName, const QString& directory);
   /**
   * changes the language to new lang and rereads the files
   */
   void setLanguage(const QString& lang);

   bool addToMOFiles(QString url);
   bool setCompendiumFile(QString url);
   bool setAuxiliaryFile(QString url);

public slots:
   /** displays next search result */
   void nextResult();
   /** displays previous search result */
   void prevResult();
   /** stops the search if we are currently searching */
   void stopSearch();

   void setSettings(SearchSettings settings);

signals:
   /** emitted when a search started or stopped */
   void signalSearchActive(bool);
   /**
   * emitted after filepaths are checked to signal
   * if it's possible to search in catalogs
   */
   void signalEnableCatSearch(bool);
   /**
   * emitted after filepaths are checked to signal
   * if it's possible to search in compendium file
   */
   void signalEnableComSearch(bool);
   /**
   * emitted after filepaths are checked to signal
   * if it's possible to search in auxiliary file
   */
   void signalEnableAuxSearch(bool);

   void signalSettingsChanged(SearchSettings);

   void signalResetProgressBar(QString,int);
   void signalProgress(int);
   void signalClearProgressBar();

protected:
   virtual void dragEnterEvent( QDragEnterEvent* event);
   virtual void dropEvent( QDropEvent* event);

private slots:
   void startDelayedSearch();

private:
   /** starts search in installed message catalogs */
   void catSearch(QString msgid);
   /** starts search in auxiliary file*/
   void auxSearch(QString msgid);
   /** starts search in compendium file*/
   void comSearch(QString msgid);

   void createCatList();
   void findAuxFile();
   void findComFile();

   /** clears everything necessary for next search */
   void clearResults();

   void showResult(int index);

   /** set total found entries to total */
   void setTotalResults(int total);


private:
   SearchMode _searchMode;
   QString _searchString;

   bool _stopSearch;
   bool _searching;

   QValueList<SearchResultItem> _results;
   int _currentResult;

   QString _packageName;
   QString _packageDir;
   QFile _auxFile;
   QFile _comFile;
   QList<KLocale> _catList;

   SearchSettings _settings;


   // gui
   QMultiLineEdit* _msgidBox;
   QMultiLineEdit* _msgstrBox;
   QPushButton* _nextButton;
   QPushButton* _prevButton;

   QLabel* _fileLabel;
   QLabel* _totalLabel;

   QPopupMenu* _textDropMenu;
   QPopupMenu* _urlDropMenu;
};


#endif // SEARCHRESULTBOX_H

