/* ****************************************************************************
  This file is part of KBabel

  Copyright (C) 1999 by Matthias Kiefer
                            <matthias.kiefer@gmx.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

**************************************************************************** */
#ifndef CATALOGMANAGER_H
#define CATALOGMANAGER_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qdict.h>
#include <qlistview.h>
#include <qdatetime.h>
#include <qtimer.h>
#include <qfileinfo.h>
#include <qguardedptr.h>

#include <kdialogbase.h>
#include <kdirwatch.h>
#include <kprocess.h>
#include <qlist.h>

#include "settings.h"

class CatManListItem;
class QPixmap;
class QPopupMenu;
class KBabel;
class KAction;
class CatManPreferences;
class QMultiLineEdit;


class CatalogManagerView : public QListView
{
   Q_OBJECT
public:
   CatalogManagerView(QWidget* parent=0, const char* name=0);
   virtual ~CatalogManagerView();

   CatManSettings settings() const;
   /** clears the tree */
   virtual void clear();

   /** pauses the update timer */
   void pause(bool);

   /** if file fileWithPath exists in the treeview, this is updated */
   void updateFile(QString fileWithPath);

   bool isActive() const {return _active;}

   void restoreView();
   void saveView() const;

public slots:
   void setSettings(CatManSettings newSettings);
   void clearAllMarks();
   void toggleAllMarks();
   /**
   * traverses all childs in the directory of the current item
   * (including all subdirectories) and displays some statistics
   * about the translations. If the item is a file, its
   * parent directory is used instead.
   */
   void statistics();
   /** calls @ref Msgfmt::checkSyntax, to check the po-file of the selected item */
   void checkSyntax();

   void showLog();

   void stop();

signals:
   void openFile(QString filename);
   void openTemplate(QString openFilename,QString saveFileName);
   void settingsChanged(CatManSettings);
   void prepareProgressBar(QString msg,int max);
   void progress(int);
   void clearProgressBar();

protected:
   /**
   * builds the tree under dir relDir, but does not update any files
   * this functions always traverses all subdirs
   *
   * @param relDir the relative dir under the po- and pot- base directories
   * @param fast if true, no files will be updated
   *
   * @return true, if the directory contains any po or pot-files
   * @see CatalogManagerView::buildDir
   * @see CatalogManagerView::updateDir
   */
   bool buildDir(QString relDir,bool fast=true);

   /**
   * This function is traversing the real directory on the disc using baseDir as the
   * base directory and starts at baseDir+relDir
   * @param extension the extension of the files in this directory
   * @param fast if true, no file will be updated
   *
   * @return true, if the directory contains any po or pot-files
   * @see CatalogManagerView::buildDir
   * @see CatalogManagerView::updateDir
   */
   bool buildDir(const QString& baseDir,const QString& relDir, const QString extension,bool fast=true);

   /**
   * updates dir relDir and if any new subdir is added builds this with @ref buildDir
   *
   * This function doesn't enters subdirs except when a new subdir is added.
   * @see CatalogManagerView::buildDir
   */
   void updateDir(QString relDir);

   /**
   * stops the update timer and the dirwatch
   * @see KDirWatch::stop
   * @see QTimer::stop
   */
   virtual void hideEvent(QHideEvent*);
   /**
   * restarts the update timer and the dirwatch
   * @see KDirWatch::start
   * @see QTimer::start
   */
   virtual void showEvent(QShowEvent*);

   /** used for dragging */
   virtual void contentsMousePressEvent(QMouseEvent* e);
   /** used for dragging */
   virtual void contentsMouseMoveEvent(QMouseEvent* e);

protected slots:
   /** rebuilds the tree*/
   void buildTree();
   /**
   * recurse all visible files and updates them if necessary
   * @see CatManListItem::checkUpdate
   */
   void checkUpdate();

   /** this is called from KDirWatch when a directory has changed */
   void directoryChanged(const QString& dir);
   /** this is called from KDirWatch when a directory has been deleted */
   void directoryDeleted(const QString& dir);

   void showContentsMenu(QListViewItem *, const QPoint &, int col);
   /** does the default action on the currently selected item*/
   void activateItem(QListViewItem *);
   /** enables or disables some actions, depending on the status of this item */
   void checkSelected(QListViewItem *);
   /** calls @ref activateItem with the selected item as argument*/
   void slotOpenFile();
   /** emits signal @ref openTemplate */
   void slotOpenTemplate();
   /** deletes the po-file on the disc, that belongs to the selected item */
   void slotDeleteFile();
   /** toggles the mark of the selected item */
   void slotToggleMark();
   /**
   * toggles all marks in directory of current item (must be a directory)
   * Does not go into subdirs
   */
   void slotToggleMarksInDir();
   /**
   * removes all marks in directory of current item (must be a directory)
   * Does not go into subdirs
   */
   void slotClearMarksInDir();

   void slotDirCommand(int);
   void slotFileCommand(int);

private slots:
   void showOutput(KProcess *proc, char *buffer, int buflen);
   void processEnded(KProcess *proc);

private:
   void readMarker();
   void saveMarker();

private:
   QDict<CatManListItem> _dirList;
   QDict<CatManListItem> _fileList;

   KDirWatch *_dirWatch;
   QTimer *_updateTimer;
   CatManSettings _settings;

   QStringList _markerList;

   bool _active;
   bool _stop;

   QList<KProcess> _pendingProcesses;

   QMultiLineEdit* _logView;
   KDialogBase* _logWindow;
   QPopupMenu* _fileContentsMenu;
   QPopupMenu* _dirContentsMenu;
   QPopupMenu* _dirCommandsMenu;
   QPopupMenu* _fileCommandsMenu;
   KAction* _openAction;
   KAction* _openTemplateAction;
   KAction* _checkAction;
   KAction* _deleteAction;
   KAction* _markAction;

   /** used for starting a drag */
   QPoint _pressPos;
};



class CatManListItem : public QListViewItem
{
public:
   /** the type of this item */
   enum Type{File,Dir};

   CatManListItem(QListViewItem* parent,QString fullPath,QString fullPotPath,QString package);
   /** creates the toplevel root item with package name "/" */
   CatManListItem(QListView* parent,QString fullPath, QString fullPotPath);

   /** returns the package names (including relative path) of the children of this item */
   QStringList contentsList(bool onlyFiles=false) const;
   /**
   * returns the package names of all children of this item (including all subdirectries)
   * @param onlyFiles flag, if only the names of files should be returned
   * @see CatManListItem::contentsList
   */
   QStringList allChildrenList(bool onlyFiles=false) const;

   void setMarked(bool on);
   bool marked() const{ return _marked;}

   /**
   * checks if the file on the disc has changed,
   * reads information about the file and displays it
   */
   void checkUpdate();


   /** return the absolute filename of the po-File */
   QString poFile() const;
   /** return the absolute filename of the pot-File */
   QString potFile() const;
   /** returns the package name (inlcuding relative path to base-directory) */
   QString package(bool rootSlash=true) const;

   /** returns the package name (without path) */
   QString name() const;

   /**
   * returns the type of this item
   * @see CatManListItem::Type
   */
   Type type() const{return _type;}
   bool isDir() const;
   bool isFile() const;
   /** returns true, if the po-file exists*/
   bool hasPo() const;
   /** returns true, if the pot-file exists*/
   bool hasPot() const;
   /** returns the number of fuzzy messages in the po-file, 0 if no po-file exists*/
   int fuzzy() const;
   /** returns the number of untranslated messages in the po-file, @ref total if no po-file exists*/
   int untranslated() const;
   /** returns number of messages in the po- or pot-file*/
   int total() const;

   virtual QString key(int col,bool) const;
   virtual void setOpen(bool);

   /** paints the marking, if this package has no template */
   static void paintExclamation(QPixmap*);

private:
   void init(const QString& fullPath, const QString& fullPotPath,const QString& package);
   /**
   * updates the item
   * @param showPoInfo if true, reads information about the file using @ref Catalog::info
   * ( slow for big files )
   * @param includeChildren flag, if possible children should be updated,too
   */
   void update(bool showPoInfo=true,bool includeChildren=false);

private:
   /**
   * holds the date and the time this item was
   * last updated. This is used to check, if the file
   * on the disc has changed since last update.
   */
   QDateTime _lastUpdated;

   /** the po-file */
   QFileInfo _primary;
   /** the pot-file */
   QFileInfo _template;
   /**
   * The package name, includes the relative path beginning at the base directory.
   * The package name begins with "/" and if this is a directory it end with "/"
   * The root item has the package name "/"
   * @see CatManListItem::CatManListItem
   */
   QString _package;
   Type _type;
   bool _marked;
};




class CatalogManager : public KDialogBase
{
   Q_OBJECT
public:
   CatalogManager(CatManSettings settings);
   ~CatalogManager();

   CatManSettings settings() const;
   /**
   * Sets the window, in which the files should be opened.
   * This is set by KBabel::openCatalogManager
   */
   void setPreferredWindow(KBabel*);
   /** updates the file fileWithPath in the @ref CatalogManagerView */
   void updateFile(QString fileWithPath);

public slots:
   void setSettings(CatManSettings newSettings);
   virtual void slotHelp();
   /** called by the configure button */
   virtual void slotUser1();

signals:
   void settingsChanged(CatManSettings);

private:
   void restoreView();
   void saveView() const;

private slots:
   /**
   * calls @ref KBabel::open where as preferred windos _preferredWindow
   * is used. If this is deleted meanwhile, the first window in
   * @ref KTMainWindow::memberList is used.
   */
   void openFile(QString filename);
   /**
   * calls @ref KBabel::openTemplate where as preferred windos _preferredWindow
   * is used. If this is deleted meanwhile, the first window in
   * @ref KTMainWindow::memberList is used.
   */
   void openTemplate(QString openFilename,QString saveFileName);

private:
   CatalogManagerView* _catalogManager;
   QGuardedPtr<KBabel> _preferredWindow;

   bool _openNewWindow;

   KDialogBase* _prefDialog;
   CatManPreferences* _prefWidget;
};

#endif // CATALOGMANAGER_H
