/* ****************************************************************************
  This file is part of KBabel

  Copyright (C) 1999 by Matthias Kiefer
                            <matthias.kiefer@gmx.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

**************************************************************************** */
#include <qtextstream.h>
#include <qfile.h>
#include <qdir.h>
#include <qfileinfo.h>
#include <qregexp.h>
#include <qstring.h>
#include <qtextcodec.h>
#include <qdatetime.h>

//#include <kconfigbase.h>
#include <kconfig.h>
#include <kglobal.h>
#include <klocale.h>
#include <kapp.h>
#include <kio_netaccess.h>
#include <kurl.h>
#include <kregexp.h>

#include "catalog.h"
#include "catalogitem.h"
#include "kbabelview.h"
#include "headereditor.h"

#include "resources.h"
#include "version.h"
#include "settings.h"


Catalog::Catalog(QObject* parent, const char* name) : QObject(parent,name)
{
   _modified=false;
   _readOnly=false;
   readPreferences();

   _headerEditor=0;

   _views.setAutoDelete(false);
}


Catalog::~Catalog()
{
   if(_headerEditor)
      delete _headerEditor;
}

QString Catalog::msgid(uint index) const
{
   uint max=_entries.count()-1;
   if(index > max)
      index=max;

   return _entries[index].msgid();
}

QString Catalog::msgstr(uint index) const
{
   uint max=_entries.count()-1;
   if(index > max)
      index=max;

   return _entries[index].msgstr();
}

QString Catalog::comment(uint index) const
{
   uint max=_entries.count()-1;
   if(index > max)
      index=max;

   return _entries[index].comment();
}

CatalogItem Catalog::header() const
{
   return _header;
}

bool Catalog::setMsgstr(uint index,QString msgstr)
{
/*
   KASSERT1((index < _entries.count()),KDEBUG_ERROR,KBABEL
         ,"Catalog::setMsgstr: wrong index: %i",index);
  */
   bool untranslatedChanged=false;

   if(_entries[index].isUntranslated() && !msgstr.isEmpty())
   {
      _untransIndex.remove(index);
      untranslatedChanged=true;
   }
   else if(msgstr.isEmpty())
   {
      QValueList<uint>::Iterator it;

      // insert index in the right place in the list
      it = _untransIndex.begin();
      while(it != _untransIndex.end() && index > (*it))
      {
         ++it;
      }
      _untransIndex.insert(it,index);

      untranslatedChanged=true;
   }

   _entries[index].setMsgstr(msgstr);

   setModified(true);

   if(untranslatedChanged)
      emit signalNumberOfUntranslatedChanged(numberOfUntranslated());

   return untranslatedChanged;
}


bool Catalog::setComment(uint index,QString comment)
{
/*
   KASSERT2((index < _entries.count()),KDEBUG_ERROR,KBABEL
         ,"Catalog::setComment: wrong index:%i : %i",index,_entries.count());
  */
   bool fuzziesChanged=false;


   bool wasFuzzy=_entries[index].isFuzzy();

   _entries[index].setComment(comment);

   bool isFuzzy=_entries[index].isFuzzy();

   if(wasFuzzy && !isFuzzy)
   {
      _fuzzyIndex.remove(index);
      fuzziesChanged=true;
   }
   else if(isFuzzy)
   {
      QValueList<uint>::Iterator it;

      // insert index in the right place in the list
      it = _fuzzyIndex.begin();
      while(it != _fuzzyIndex.end() && index > (*it))
      {
         ++it;
      }
      _fuzzyIndex.insert(it,index);

      fuzziesChanged=true;
   }

   setModified(true);

   if(fuzziesChanged)
      emit signalNumberOfFuzziesChanged(numberOfFuzzies());


   return fuzziesChanged;
}

bool Catalog::setHeader(CatalogItem newHeader)
{
   if(newHeader.isValid())
   {
      _header=newHeader;
      setModified(true);
      return true;
   }

   return false;
}

QString Catalog::currentURL() const
{
   if(_url.isLocalFile())
   {
      return _url.path(0);
   }
   else
      return _url.url();
}

void Catalog::setCurrentURL(QString url)
{
   _url=KURL(QDir::currentDirPath()+"/",url);
}


CatalogItem Catalog::updatedHeader(CatalogItem oldHeader, bool usePrefs) const
{
   QStringList headerList=oldHeader.msgstrAsList();

   QStringList::Iterator it;
   QString temp;
   bool found;
   if(!usePrefs || _saveSettings.updateLastTranslator)
   {
      found=false;

      temp="Last-Translator: "+_identitySettings.authorName;
      if(!_identitySettings.authorEmail.isEmpty())
      {
         temp+=(" <"+_identitySettings.authorEmail+">");
      }
      temp+="\\n";
      for( it = headerList.begin(); it != headerList.end(); ++it )
      {
         if((*it).contains(QRegExp("^ *Last-Translator:.*")))
         {
            (*it).replace(QRegExp("^ *Last-Translator:.*"),temp);
            found=true;
         }
       }
       if(!found)
       {
          headerList.append(temp);
       }
   }
   if(!usePrefs || _saveSettings.updateRevisionDate)
   {
      found=false;

      temp="PO-Revision-Date: "+dateTime()+"\\n";

      for( it = headerList.begin(); it != headerList.end(); ++it )
      {
         if((*it).contains(QRegExp("^ *PO-Revision-Date:.*")))
         {
            (*it).replace(QRegExp("^ *PO-Revision-Date:.*"),temp);
            found=true;
         }
       }
       if(!found)
       {
          headerList.append(temp);
       }
   }
   if(!usePrefs || _saveSettings.updateLanguageTeam)
   {
      found=false;

      temp="Language-Team: "+_identitySettings.languageName;
      if(!_identitySettings.mailingList.isEmpty())
      {
         temp+=(" <"+_identitySettings.mailingList+">");
      }
      temp+="\\n";
      for( it = headerList.begin(); it != headerList.end(); ++it )
      {
         if((*it).contains(QRegExp("^ *Language-Team:.*")))
         {
            (*it).replace(QRegExp("^ *Language-Team:.*"),temp);
            found=true;
         }
       }
       if(!found)
       {
          headerList.append(temp);
       }
   }
   if(!usePrefs || _saveSettings.updateCharset)
   {
      found=false;

      QString encodingStr;
      switch(_saveSettings.encoding)
      {
         case UTF8:
            encodingStr=QTextCodec::codecForName("UTF-8")->name();
            break;
         case UTF16:
            encodingStr=QTextCodec::codecForName("UTF-16")->name();
            break;
         default:
            encodingStr=QTextCodec::codecForLocale()->name();
      }
      temp="Content-Type: text/plain; charset="+encodingStr+"\\n";

      for( it = headerList.begin(); it != headerList.end(); ++it )
      {
         if((*it).contains(QRegExp("^ *Content-Type:.*")))
         {
            (*it).replace(QRegExp("^ *Content-Type:.*"),temp);
            found=true;
         }
       }
       if(!found)
       {
          headerList.append(temp);
       }
   }
   if(!usePrefs || _saveSettings.updateEncoding)
   {
      found=false;

      temp="Content-Transfer-Encoding: 8bit\\n";

      for( it = headerList.begin(); it != headerList.end(); ++it )
      {
         if((*it).contains(QRegExp("^ *Content-Transfer-Encoding:.*")))
         {
            (*it).replace(QRegExp("^ *Content-Transfer-Encoding:.*"),temp);
            found=true;
         }
       }
       if(!found)
       {
          headerList.append(temp);
       }
   }

   temp="X-Generator: KBabel %1\\n";
   temp=temp.arg(VERSION);
   found=false;

   for( it = headerList.begin(); it != headerList.end(); ++it )
   {
      if((*it).contains(QRegExp("^ *X-Generator:.*")))
      {
         (*it).replace(QRegExp("^ *X-Generator:.*"),temp);
         found=true;
      }
    }
    if(!found)
    {
       headerList.append(temp);
    }

   QString msgstr;
   for( it = headerList.begin(); it != headerList.end(); ++it )
   {
      msgstr+=("\n"+(*it));
   }

   msgstr.remove(0,1);// remove first newline

   oldHeader.setMsgstr(msgstr);
   return oldHeader;
}


void Catalog::removeFuzzyStatus(uint index)
{
   uint max=_entries.count()-1;
   if(index > max)
      index=max;

   if(_entries[index].isFuzzy())
   {
      _entries[index].removeFuzzy();
      setModified(true);
      _fuzzyIndex.remove(index);

      emit signalNumberOfFuzziesChanged(numberOfFuzzies());
   }
}


void Catalog::setModified(bool flag)
{
    bool old=_modified;
    _modified=flag;

    if(old!=_modified);
       emit signalModified(flag);
}


QString Catalog::packageName() const
{
    QString package=_url.filename();

    int index=package.findRev(QRegExp(".pot?"));

    if(index>0)
      package=package.left(index);

    return package;
}

QString Catalog::packageDir() const
{
    QString dir=_url.directory();

    return dir;
}


Catalog::IOStatus Catalog::openURL(QString url)
{
   QString target;

   KURL tempURL(QDir::currentDirPath()+"/",url);

   if(KIONetAccess::download(tempURL.url(), target))
   {
        // load in the file (target is always local)
        IOStatus success=openFile(target);

        // and remove the temp file
        KIONetAccess::removeTempFile(target);

        // store current url
        if(success==OK)
        {
           setModified(false);
           _url=tempURL;

           emit signalFileOpened(_readOnly);
           emit signalNumberOfFuzziesChanged(numberOfFuzzies());
           emit signalNumberOfUntranslatedChanged(numberOfUntranslated());
           emit signalTotalNumberChanged(numberOfEntries());
        }

        return success;
   }
   else
   {
      return OS_ERROR;
   }
}

Catalog::IOStatus Catalog::openURL(QString openURL,QString saveURL)
{
   QString target;

   KURL tempURL(QDir::currentDirPath()+"/",openURL);

   if(KIONetAccess::download(tempURL.url(), target))
   {
        // load in the file (target is always local)
        IOStatus success=openFile(target);

        // and remove the temp file
        KIONetAccess::removeTempFile(target);

        // store current url
        if(success==OK)
        {
           setModified(false);
           _url=KURL(QDir::currentDirPath()+"/",saveURL);

           emit signalFileOpened(_readOnly);
           emit signalNumberOfFuzziesChanged(numberOfFuzzies());
           emit signalNumberOfUntranslatedChanged(numberOfUntranslated());
           emit signalTotalNumberChanged(numberOfEntries());
        }

        return success;
   }
   else
   {
      return OS_ERROR;
   }
}

Msgfmt::Status Catalog::checkSyntax(QString& output)
{
   QString filename;
   bool tempFileUsed=false;

   if(_url.isLocalFile() && !isModified())
   {
      filename=_url.path(0);
   }
   else
   {
      tempFileUsed=true;
      filename=saveTempFile();
   }

   Msgfmt msgfmt;
   Msgfmt::Status result = msgfmt.checkSyntax( filename , output );

   _errorIndex.clear();

   if( result==Msgfmt::SyntaxError )
   {
      int currentIndex=-1;
      int currentLine=0;

      if( !_header.msgstr().isEmpty() )
         currentLine=_header.totalLines()+1;

      QStringList lines = QStringList::split("\n",output);
      for ( QStringList::Iterator it = lines.begin(); it != lines.end(); ++it )
      {
         if( (*it).contains(QRegExp("^.+:\\d+:")) )
         {
            int begin=(*it).find(":",0)+1;
            int end=(*it).find(":",begin);

            QString line=(*it).mid(begin,end-begin);

            while( line.toInt() > currentLine )
            {
               currentIndex++;
               currentLine += ( _entries[currentIndex].totalLines() + 1 );
            }

            if( !_errorIndex.contains(currentIndex) )
            {
               _errorIndex.append(currentIndex);
            }
         }
      }
   }

   if(tempFileUsed)
      QFile::remove(filename);

   return result;
}

void Catalog::clear()
{
    _entries.clear();
    _url=KURL();
}


uint Catalog::numberOfEntries() const
{
   return _entries.count();
}

uint Catalog::numberOfFuzzies() const
{
   return _fuzzyIndex.count();
}

uint Catalog::numberOfUntranslated() const
{
   return _untransIndex.count();
}


bool Catalog::hasFuzzyInFront(uint index)  const
{
   if(findPrevInList(_fuzzyIndex,index)>=0)
   {
      return true;
   }

   return false;
}

bool Catalog::hasFuzzyAfterwards(uint index) const
{
   if(findNextInList(_fuzzyIndex,index)>=0)
   {
      return true;
   }

   return false;
}

bool Catalog::hasUntranslatedInFront(uint index) const
{
   if(findPrevInList(_untransIndex,index)>=0)
   {
      return true;
   }

   return false;
}

bool Catalog::hasUntranslatedAfterwards(uint index) const
{
   if(findNextInList(_untransIndex,index)>=0)
   {
      return true;
   }

   return false;
}

bool Catalog::hasErrorInFront(uint index)  const
{
   if(findPrevInList(_errorIndex,index)>=0)
   {
      return true;
   }

   return false;
}

bool Catalog::hasErrorAfterwards(uint index) const
{
   if(findNextInList(_errorIndex,index)>=0)
   {
      return true;
   }

   return false;
}

bool Catalog::isFuzzy(uint index) const
{
   if(index > numberOfEntries())
      return false;

   return _entries[index].isFuzzy();
}


bool Catalog::isUntranslated(uint index) const
{
   if(index > numberOfEntries())
      return false;

   return _entries[index].isUntranslated();
}

bool Catalog::hasError(uint index) const
{
   return _errorIndex.contains(index);
}


int Catalog::nextFuzzy(uint startIndex) const
{
   return findNextInList(_fuzzyIndex,startIndex);
}

int Catalog::prevFuzzy(uint startIndex) const
{
   return findPrevInList(_fuzzyIndex,startIndex);
}

int Catalog::nextUntranslated(uint startIndex) const
{
   return findNextInList(_untransIndex,startIndex);
}

int Catalog::prevUntranslated(uint startIndex) const
{
   return findPrevInList(_untransIndex,startIndex);
}


int Catalog::nextError(uint startIndex) const
{
   return findNextInList(_errorIndex,startIndex);
}

int Catalog::prevError(uint startIndex) const
{
   return findPrevInList(_errorIndex,startIndex);
}


void Catalog::registerView(KBabelView* view)
{
   if(_views.containsRef(view)==0)
   {
      _views.append(view);
   }
}


void Catalog::removeView(KBabelView* view)
{
   _views.removeRef(view);
}


void Catalog::updateViews(int index,KBabelView* view2exclude)
{
    KBabelView* view;
    for ( view=_views.first(); view != 0; view=_views.next())
    {
       if(view!=view2exclude)
       {
          view->updateEntry(index);
       }
    }
}



bool Catalog::hasView() const
{
    if(_views.count()==0)
           return false;

    return true;
}

bool Catalog::isLastView() const
{
    if(_views.count()<=1)
           return true;

    return false;
}


HeaderEditor* Catalog::headerEditor()
{
   if(!_headerEditor)
   {
      _headerEditor = new HeaderEditor(this,"_headerEditor");
   }

   return _headerEditor;
}


void Catalog::readPreferences()
{
   //KConfigBase* config=KGlobal::config();
   KConfig* config=KGlobal::config();

   KConfigGroupSaver groupSaver(config,"Header");

   _saveSettings.autoUpdate=config->readBoolEntry("AutoUpdate",Defaults::Save::autoUpdate);
   _saveSettings.updateLastTranslator=config->readBoolEntry("Update-Last-Translator"
                       ,Defaults::Save::updateLastTranslator);
   _saveSettings.updateRevisionDate=config->readBoolEntry("Update-Revision-Date"
                       ,Defaults::Save::updateRevisionDate);
   _saveSettings.updateLanguageTeam=config->readBoolEntry("Update-Language-Team"
                       ,Defaults::Save::updateLanguageTeam);
   _saveSettings.updateCharset=config->readBoolEntry("Update-Charset"
                       ,Defaults::Save::updateCharset);
   _saveSettings.updateEncoding=config->readBoolEntry("Update-Encoding"
                       ,Defaults::Save::updateEncoding);
   _saveSettings.encoding=(Encoding)(config->readNumEntry("Encoding",(int)Defaults::Save::encoding));

   _saveSettings.autoSyntaxCheck = config->readBoolEntry("AutoSyntaxCheck"
                 ,Defaults::Save::autoSyntaxCheck);
   _saveSettings.customDateFormat = config->readEntry("CustomDateFormat"
                 ,Defaults::Save::customDateFormat);
   _saveSettings.dateFormat = (DateFormat)( config->readNumEntry("DateFormat"
                 ,(int)Defaults::Save::dateFormat) );

   _identitySettings.authorName=config->readEntry("Author-Name",Defaults::Identity::authorName());
   _identitySettings.authorEmail=config->readEntry("Author-Email",Defaults::Identity::authorEmail());
   _identitySettings.languageName=config->readEntry("Language",Defaults::Identity::languageName());
   _identitySettings.mailingList=config->readEntry("Mailinglist",Defaults::Identity::mailingList());
   _identitySettings.timeZone=config->readEntry("Timezone",Defaults::Identity::timezone());
}

void Catalog::savePreferences()
{
   //KConfigBase* config=KGlobal::config();
   KConfig* config=KGlobal::config();

   KConfigGroupSaver groupSaver(config,"Header");

   config->writeEntry("AutoUpdate",_saveSettings.autoUpdate);
   config->writeEntry("Update-Last-Translator",_saveSettings.updateLastTranslator);
   config->writeEntry("Update-Revision-Date",_saveSettings.updateRevisionDate);
   config->writeEntry("Update-Language-Team",_saveSettings.updateLanguageTeam);
   config->writeEntry("Update-Charset",_saveSettings.updateCharset);
   config->writeEntry("Update-Encoding",_saveSettings.updateEncoding);
   config->writeEntry("Encoding",(int)_saveSettings.encoding);

   config->writeEntry("AutoSyntaxCheck",_saveSettings.autoSyntaxCheck);
   config->writeEntry("CustomDateFormat",_saveSettings.customDateFormat);
   config->writeEntry("DateFormat",(int)_saveSettings.dateFormat);

   config->writeEntry("Author-Name",_identitySettings.authorName);
   config->writeEntry("Author-Email",_identitySettings.authorEmail);
   config->writeEntry("Language",_identitySettings.languageName);
   config->writeEntry("Mailinglist",_identitySettings.mailingList);
   config->writeEntry("Timezone",_identitySettings.timeZone);
}


void Catalog::setSettings(SaveSettings settings)
{
   _saveSettings=settings;

   emit signalSettingsChanged(settings);
}

void Catalog::setSettings(IdentitySettings settings)
{
   _identitySettings=settings;

   emit signalSettingsChanged(settings);
}


void Catalog::generateIndexLists()
{
   _fuzzyIndex.clear();
   _untransIndex.clear();
   _errorIndex.clear();

   uint counter=0;
   for ( QValueList<CatalogItem>::Iterator it = _entries.begin(); it != _entries.end(); ++it )
   {
       if((*it).isUntranslated())
       {
          _untransIndex.append(counter);
       }
       else if((*it).isFuzzy())
       {
          _fuzzyIndex.append(counter);
       }

       counter++;
   }

}

int Catalog::findNextInList(const QValueList<uint>& list,uint index) const
{
    QValueList<uint>::ConstIterator it;

    int nextIndex=-1;

    // find index in List
    it=list.find(index);

    // if the given index is found in the list and not the last entry
    // in the list, return the next listentry
    if(it!=list.end() && it!=list.fromLast())
    {
       ++it;
       return (*it);
    }

    // if the index is not in the list, search the index in the list, that
    // is the nearest to the given index
    for( it = list.begin(); it != list.end(); ++it )
    {
       if((*it) > index)
       {
          nextIndex=(*it);
          break;
       }
    }


    return nextIndex;
}

int Catalog::findPrevInList(const QValueList<uint>& list,uint index) const
{
    QValueList<uint>::ConstIterator it;

    int prevIndex=-1;

    it=list.find(index);

    // if the given index is found in the list and not the last entry
    // in the list, return the next listentry
    if(it!=list.end() && it!=list.begin())
    {
       --it;
       return (*it);
    }


    // if the index is not in the list, search the index in the list, that
    // is the nearest to the given index
    for( it = list.fromLast(); it != list.end(); --it )
    {
       if((*it) < index)
       {
          prevIndex=(*it);
          break;
       }
    }


    return prevIndex;
}


QString Catalog::dateTime() const
{
    QString dateTimeString;
    QDate date=QDate::currentDate();
    QTime time=QTime::currentTime();

    switch(_saveSettings.dateFormat)
    {
       case Local:
       {
          return KGlobal::locale()->formatDateTime(QDateTime::currentDateTime());
       }
       case Default:
          dateTimeString = Defaults::Save::customDateFormat;
          break;
       case Custom:
          dateTimeString = _saveSettings.customDateFormat;
          break;
    }

    // the year
    dateTimeString.replace( QRegExp("%Y"), QString::number( date.year() ) );
    dateTimeString.replace( QRegExp("%y"), QString::number( date.year() ).right(2) );

    // the month
    if(date.month()<10)
    {
       dateTimeString.replace( QRegExp("%m"), "0"+QString::number( date.month() ) );
    }
    else
    {
       dateTimeString.replace( QRegExp("%m"), QString::number( date.month() ) );
    }

    dateTimeString.replace( QRegExp("%f"), QString::number( date.month() ) );

    dateTimeString.replace( QRegExp("%b"), date.monthName(date.month()) );
    dateTimeString.replace( QRegExp("%h"), date.monthName(date.month()) );

    // the day
    dateTimeString.replace( QRegExp("%j"), QString::number( date.dayOfYear() ) );
    dateTimeString.replace( QRegExp("%e"), QString::number( date.day() ) );
    if(date.day() < 10)
    {
       dateTimeString.replace( QRegExp("%d"), "0"+QString::number( date.day() ) );
    }
    else
    {
       dateTimeString.replace( QRegExp("%d"), QString::number( date.day() ) );
    }

    dateTimeString.replace( QRegExp("%a"), date.dayName( date.dayOfWeek() ) );


    // hour
    dateTimeString.replace( QRegExp("%k"), QString::number( time.hour() ) );

    if(time.hour() < 10)
    {
       dateTimeString.replace( QRegExp("%H"), "0"+QString::number( time.hour() ) );
    }
    else
    {
       dateTimeString.replace( QRegExp("%H"), QString::number( time.hour() ) );
    }

    QString zone;
    int hour;
    if( time.hour() > 12 )
    {
       zone="PM";
       hour=time.hour()-12;
    }
    else
    {
       zone="AM";
       hour=time.hour();
    }

    dateTimeString.replace( QRegExp("%I"), QString::number( hour ) );

    if(hour < 10)
    {
       dateTimeString.replace( QRegExp("%i"), "0"+QString::number( hour ) );
    }
    else
    {
       dateTimeString.replace( QRegExp("%i"), QString::number( hour ) );
    }

    dateTimeString.replace( QRegExp("%p"), zone );

    // minutes
    if(time.minute() < 10)
    {
       dateTimeString.replace( QRegExp("%M"), "0"+QString::number( time.minute() ) );
    }
    else
    {
       dateTimeString.replace( QRegExp("%M"), QString::number( time.minute() ) );
    }

    // seconds
    if(time.second() < 10)
    {
       dateTimeString.replace( QRegExp("%S"), "0"+QString::number( time.second() ) );
    }
    else
    {
       dateTimeString.replace( QRegExp("%S"), QString::number( time.second() ) );
    }

    // timezone
    dateTimeString.replace( QRegExp("%Z"), _identitySettings.timeZone );
    dateTimeString.replace( QRegExp("%z"), _identitySettings.timeZone );

    return dateTimeString;
}


Catalog::IOStatus Catalog::readHeader(QTextStream& stream, CatalogItem& header)
{
   CatalogItem temp;
   int filePos=stream.device()->at();
   CatalogItem::IOStatus status=temp.read(stream);

   if(status==CatalogItem::Ok)
   {
      // test if this is the header
      if(temp.msgid().isEmpty())
      {
          header=temp;
          if(header.isFuzzy())
          {
             header.removeFuzzy();
          }
      }
      else
      {
         stream.device()->at(filePos);
      }

      return OK;
   }

   return PARSE_ERROR;

}

Catalog::IOStatus Catalog::openFile(QString filename)
{
   KASSERT(!filename.isEmpty(),KDEBUG_FATAL,KBABEL
           , "fatal error: empty filename to open");

   QFileInfo info(filename);

   if(!info.exists() || info.isDir())
      return NO_FILE;

   if(!info.isReadable())
      return NO_PERMISSIONS;

   QFile file(filename);


   if(file.open(IO_ReadOnly))
   {
      emit signalResetProgressBar(i18n("loading file"),file.size());

      // find codec for file
      QTextCodec* codec=codecForFile(file);

      QTextStream stream(&file);

      if(codec)
         stream.setCodec(codec);


      // if somethings goes wrong with the parsing, we don't have deleted the old contents
      QValueList<CatalogItem> tempEntries;
      CatalogItem tempHeader;


      kDebugArea(KBABEL,"start parsing...");

      // first read header
      IOStatus status = readHeader(stream,tempHeader);
      if(status!=OK)
      {
          emit signalClearProgressBar();

          file.close();
          return status;
      }

      // now parse the rest of the file

      CatalogItem tempCatItem;
      CatalogItem::IOStatus success=CatalogItem::Ok;

      while(!stream.eof() && success==CatalogItem::Ok)
      {
         kapp->processEvents();

         success=tempCatItem.read(stream);

         if(success==CatalogItem::Ok)
         {
            // add new entry to the list of entries
            tempEntries.append(tempCatItem);
            tempCatItem.clear();
         }

         if(file.at()%100==0)
            emit signalProgress(file.at());
      }

      file.close();


      emit signalClearProgressBar();

      kDebugArea(KBABEL,"ready.");


      if(success!=CatalogItem::ParseError)
      {
         // check if file is readOnly
         QFileInfo fi(file);
         _readOnly=!fi.isWritable();

         _entries=tempEntries;
         _header=tempHeader;

         generateIndexLists();
      }
      else
      {
         return PARSE_ERROR;
      }
   }
   else
   {
      return NO_PERMISSIONS;
   }

   return OK;
}

Catalog::IOStatus Catalog::saveFile()
{
   KASSERT(!_url.url().isEmpty(),KDEBUG_FATAL,KBABEL
          ,"fatal error: empty filename");

   if(_url.url().isEmpty())
   {
      return NO_FILE;
   }

   return saveFileAs(_url.url(),true);
}

Catalog::IOStatus Catalog::saveFileAs(QString url, bool overwrite)
{
   IOStatus status=OK;

   bool newName=false;
   KURL targetURL=_url;

   if(url != _url.url())
   {
      newName = true;
      targetURL = KURL(QDir::currentDirPath()+"/",url);
   }


   if(_saveSettings.autoUpdate)
   {
      _header=updatedHeader(_header);
   }

   if(targetURL.isLocalFile())
   {
      status=writeFile(targetURL.path(0),overwrite);
   }
   else
   {
      QString tempFile=kapp->tempSaveName(targetURL.path(0));

      status = writeFile(tempFile,overwrite);

      if(status == OK)
      {
         KURL temp(tempFile);
         if( !KIONetAccess::upload( temp.url(), targetURL.url() ) )
         {
            status = OS_ERROR;
         }
      }

      QFile::remove(tempFile);
   }

   if(status == OK)
   {
      setModified(false);

      if(newName)
      {
         // if we saved a file, the catalog can not be any longer readOnly;
         _readOnly=false;

         _url=targetURL;

         emit signalFileOpened(_readOnly);
      }
   }

   return status;
}

QString Catalog::saveTempFile()
{
   QString filename = kapp->tempSaveName("/temp/kbabel_temp.po");
   if( writeFile(filename) != OK )
   {
      filename = QString::null;
   }

   return filename;
}


Catalog::IOStatus Catalog::writeFile(QString localFile , bool overwrite)
{
   QFileInfo info(localFile);

   if(info.isDir())
      return NO_FILE;

   if(info.exists())
   {
      if(!overwrite || !info.isWritable())
      {
         return NO_PERMISSIONS;
      }
   }
   else // check if the directory is writable
   {
      QFileInfo dir(info.dirPath());
      if(!dir.isWritable())
      {
         return NO_PERMISSIONS;
      }
   }

   QFile file(localFile);

   if(file.open(IO_WriteOnly))
   {
      emit signalResetProgressBar(i18n("saving file"),numberOfEntries());

      QTextStream stream(&file);
      switch(_saveSettings.encoding)
      {
         case UTF8:
            stream.setCodec(QTextCodec::codecForName("utf-8"));
            break;
         case UTF16:
            stream.setEncoding(QTextStream::Unicode);
            break;
         default:
            break;
      }

      // only save header if it is not empty
      if(!_header.comment().isEmpty() || !_header.msgstr().isEmpty())
      {
         _header.write(stream);
         stream << "\n";
      }

      QValueList<CatalogItem>::ConstIterator it;

      int counter=1;
      QStringList list;
      for( it = _entries.begin(); it != _entries.end(); ++it )
      {
          if(counter%10==0)
             emit signalProgress(counter);

          counter++;

          (*it).write(stream);
          stream << "\n";

          kapp->processEvents();
      }

      file.close();

      emit signalClearProgressBar();
   }
   else
   {
      //emit signalError(i18n("Wasn't able to open file %1").arg(filename.ascii()));
      return OS_ERROR;
   }

   return OK;
}

QTextCodec* Catalog::codecForFile(QFile& file)
{
   bool wasOpen=true;
   int fileIndex=0;

   if(!file.isOpen())
   {
      wasOpen=false;

      if(!file.open(IO_ReadOnly))
      {
         kDebugArea(KBABEL,"wasn't able to open file");
         return 0;
      }
   }
   else
   {
      fileIndex=file.at();
   }

   QTextStream stream(&file);
   CatalogItem tempHeader;

   // first read header
   IOStatus status = readHeader(stream,tempHeader);
   if(status!=OK)
   {
       kDebugArea(KBABEL,"wasn't able to read header");
       if(!wasOpen)
          file.close();
       return 0;
   }

   QString charset;

   QString head = tempHeader.msgstr();

   KRegExp regexp("Content-Type: *text/plain; *charset *= *([^\\\"]+)");
   if( regexp.match( head.latin1() ) )
   {
       charset = regexp.group(1);
       kDebugStringArea(KBABEL,(QString("charset: ")+charset));
   }

   QTextCodec* codec=0;

   if(!charset.isEmpty())
   {
      codec=QTextCodec::codecForName(charset);

      if(!codec)
      {
         kDebugWarning("charset found, but no codec available");
      }
   }

   if(!wasOpen)
   {
       file.close();
   }
   else
   {
       file.at(fileIndex);
   }

   return codec;
}

PoInfo Catalog::headerInfo(const CatalogItem headerItem)
{
   QStringList header=headerItem.msgstrAsList();

   QStringList::Iterator it;

   PoInfo info;

      // extract information from the header
   for(it=header.begin();it!=header.end();++it)
   {
      if((*it).contains(QRegExp("^\\s*Project-Id-Version\\s*:\\s*.+\\s*$")))
      {
         info.project=(*it).replace(QRegExp("^\\s*Project-Id-Version\\s*:\\s*"),"");

         if(info.project.right(2)=="\\n")
            info.project.remove(info.project.length()-2,2);

         info.project=info.project.stripWhiteSpace();
      }
      else if((*it).contains(QRegExp("^\\s*PO-Revision-Date\\s*:\\s*.+\\s*$")))
      {
         info.revision=(*it).replace(QRegExp("^\\s*PO-Revision-Date\\s*:\\s*"),"");

         if(info.revision.right(2)=="\\n")
            info.revision.remove(info.revision.length()-2,2);

         info.revision=info.revision.stripWhiteSpace();
      }
      else if((*it).contains(QRegExp("^\\s*Last-Translator\\s*:\\s*.+\\s*$")))
      {
         info.lastTranslator=(*it).replace(QRegExp("^\\s*Last-Translator\\s*:\\s*"),"");

         if(info.lastTranslator.right(2)=="\\n")
            info.lastTranslator.remove(info.lastTranslator.length()-2,2);

         info.lastTranslator=info.lastTranslator.stripWhiteSpace();
      }
      else if((*it).contains(QRegExp("^\\s*Language-Team\\s*:\\s*.+\\s*")))
      {
         info.languageTeam=(*it).replace(QRegExp("^\\s*Language-Team\\s*:\\s*"),"");

         if(info.languageTeam.right(2)=="\\n")
            info.languageTeam.remove(info.languageTeam.length()-2,2);

         info.languageTeam=info.languageTeam.stripWhiteSpace();
      }
   }

   return info;
}


Catalog::IOStatus Catalog::info(const QString url, PoInfo& info)
{
   QString target;
   if(KIONetAccess::download(url, target))
   {
       QFile file(target);


       if(file.open(IO_ReadOnly))
       {
           QTextStream stream(&file);
           CatalogItem temp;

           // first read header
           IOStatus status = readHeader(stream,temp);
           if(status!=OK)
           {
              file.close();
              return status;
           }

           info=Catalog::headerInfo(temp);

           info.total=0;
           info.fuzzy=0;
           info.untranslated=0;

           // now parse the rest of the file
           CatalogItem::IOStatus success=CatalogItem::Ok;



           while(!stream.eof() && success==CatalogItem::Ok)
           {
               kapp->processEvents();

               success=temp.read(stream);

               if(success==CatalogItem::Ok)
               {
                  info.total++;

                  if(temp.isFuzzy())
                     info.fuzzy++;
                  else if(temp.isUntranslated())
                     info.untranslated++;
               }

           }

           file.close();

           if(success==CatalogItem::ParseError)
           {
               return PARSE_ERROR;
           }
       }
       else
       {
          return NO_PERMISSIONS;
       }


        // and remove the temp file
        KIONetAccess::removeTempFile(target);


        return OK;
   }
   else
   {
      return OS_ERROR;
   }

   return OK;
}
