/*
    SPDX-FileCopyrightText: 2022 Alexander Lohnau <alexander.lohnau@gmx.de>
    SPDX-FileCopyrightText: 2023 ivan tkachenko <me@ratijas.tk>

    SPDX-License-Identifier: LGPL-2.0-or-later
*/

pragma ComponentBehavior: Bound

import QtQuick
import QtQuick.Controls as QQC2
import QtQuick.Layouts
import QtQuick.Templates as T

import org.kde.kirigami as Kirigami
import org.kde.kcmutils as KCM

/// @since 6.0, this got renamed from KPluginDelegate to PluginDelegate
Kirigami.CheckSubtitleDelegate {
    id: listItem

    required property var model

    property list<T.Action> additionalActions

    readonly property bool enabledByDefault: model.enabledByDefault
    readonly property var metaData: model.metaData
    readonly property bool configureVisible: model.config.isValid

    signal configTriggered()

    // Let Optional chaining (?.) operator fall back to `undefined` which resets the width to an implicit value.
    width: ListView.view?.width

    icon.name: model.icon
    text: model.name
    subtitle: model.description
    checked: model.enabled

    // TODO: It should be possible to disable this
    onToggled: model.enabled = checked

    contentItem: RowLayout {
        spacing: Kirigami.Units.smallSpacing

        // Used by CheckSubtitleDelegate through duck-typing
        readonly property alias truncated: titleSubtitle.truncated

        Kirigami.IconTitleSubtitle {
            id: titleSubtitle

            Layout.fillWidth: true
            Layout.maximumWidth: implicitWidth

            icon: icon.fromControlsIcon(listItem.icon)
            title: listItem.text
            subtitle: listItem.subtitle
            reserveSpaceForSubtitle: true
        }

        Kirigami.ActionToolBar {
            Layout.fillWidth: true
            Layout.alignment: Qt.AlignRight
            alignment: Qt.AlignRight
            actions: [infoAction, configureAction, ...listItem.additionalActions]
        }
    }

    KCM.SettingHighlighter {
        target: listItem.indicator
        highlight: listItem.checked !== listItem.enabledByDefault
    }

    // Take care of displaying the actions
    readonly property Kirigami.Action __infoAction: Kirigami.Action {
        id: infoAction

        icon.name: "dialog-information"
        text: i18nc("@info:tooltip", "About")
        displayHint: Kirigami.DisplayHint.IconOnly
        onTriggered: {
            const aboutDialog = (listItem.ListView.view ?? listItem.parent.ListView.view).__aboutDialog;
            aboutDialog.metaDataInfo = listItem.metaData;
            aboutDialog.open();
        }
    }

    readonly property Kirigami.Action __configureAction: Kirigami.Action {
        id: configureAction

        visible: listItem.configureVisible
        enabled: listItem.checked
        icon.name: "configure"
        text: i18nc("@info:tooltip", "Configure…")
        displayHint: Kirigami.DisplayHint.IconOnly
        onTriggered: listItem.configTriggered()
    }
}
