/***************************************************************************
                          kpl.cpp  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sat Apr 24 15:14:00 MEST 1999

    copyright            : (C) 2002 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <math.h>
#include <qscrollbar.h>
#include <qimage.h>
#include <qdir.h>
#include <qtimer.h>
#include <qcursor.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kfiledialog.h>
#include <kprocess.h>
#include <kkeydialog.h>
#include <kmenubar.h>
#include <kstatusbar.h>
#include <kaccel.h>
#include <kconfig.h>
#include <kapp.h>
#include <kmessagebox.h>
#include <kstddirs.h>
#include <kaction.h>
#include <kstdaction.h>
#include <kedittoolbar.h>

#if KDE_VERSION == 303
#include <khelpmenu.h>
#include <kaboutdata.h>
#endif

#if (KDE_VERSION_MAJOR > 2) || (KDE_VERSION_MINOR > 1)
#include <kprinter.h>
#else
#include <qprinter.h>
#endif
#include "kpl.h"
#include "kpldoc.h"
#include "kplview.h"
#include "zoomdlg.h"
#include "refreshdlg.h"
#include "basicdlg.h"
#include "autodlg.h"
#include "itemdlg.h"
#include "autofitdlg.h"
#include "fitdlg.h"
#include "splfitdlg.h"
#include "psgraph.h"
#include "frameitem.h"
#include "arrayitem.h"
#include "funitem.h"
#include "legenditem.h"
#include "lineitem.h"
#include "arcitem.h"
#include "textitem.h"
#include "ellipseitem.h"
#include "splineitem.h"
#include "threeditem.h"
#include "utils.h"

KplApp::KplApp(QWidget*, const char* _name) :
 KMainWindow(0, _name), DCOPObject("KplIface"), plot(false), tRefresh(5)
{
  config = kapp->config();
  itemDlg = 0;
  setMinimumSize(120, 80);
  initDocument();
  initActions();
  readOptions();
  initView();
  initKeyAccel();
  calcStatusBar();
  initStatusBar();
  updWin();
  timer = new QTimer(this);
  connect(timer, SIGNAL(timeout()), SLOT(slotWatchFile()));
  connect(kapp, SIGNAL(lastWindowClosed()), kapp, SLOT(quit()));
  connect(doc, SIGNAL(modelChanged(bool, bool)), SLOT(slotUpdate(bool, bool)));
  connect(view, SIGNAL(mouseMoved(const QPoint&, bool)),
          SLOT(updStatus(const QPoint&, bool)));
  connect(view, SIGNAL(mouseLeaved()), SLOT(clearStatus()));
  connect(view, SIGNAL(rightButtonPressed()), SLOT(showPopupMenu()));
  connect(view, SIGNAL(urlsDropped(KURL::List*)),
          SLOT(slotOpenList(KURL::List*)));
}

KplApp::~KplApp()
{
  delete key_accel;
  delete itemDlg;
}

KplDoc* KplApp::getDocument() const
{
  return doc;
}

void KplApp::openList(KURL::List* list, int type)
{
  statusBar()->message(i18n("Opening file..."));
  for (unsigned i = 0; i < list->count(); i++)
    if ((!i) || doc->aut.addData) {
      if (doc->aut.addData || (warning->isChecked() ? queryClose() : true))
        doc->openDocument((*list)[i], type);
    } else {
      KplApp* kpl = new KplApp;
      if (kpl->doc->openDocument((*list)[i], type))
        kpl->show();
      else
        delete kpl;
    }
  slotStatusMsg(i18n("Ready."));
}

void KplApp::openDocumentURL(const KURL& url)
{
  statusBar()->message(i18n("Opening file..."));
  if (doc->aut.addData || (warning->isChecked() ? queryClose() : true))
    doc->openDocument(url);
  slotStatusMsg(i18n("Ready."));
}

void KplApp::slotOpenList(KURL::List* list)
{
  openList(list);
}

void KplApp::initActions()
{
  actionCollection()->setHighlightingEnabled(true);
  connect(actionCollection(), SIGNAL(actionStatusText(const QString&)),
          this, SLOT(slotStatusHelpMsg(const QString&)));
  KAction* a = KStdAction::openNew(this, SLOT(slotFileNew()),
                                   actionCollection());
  a->setStatusText(i18n("Creates a new plot"));
  a = new KAction(i18n("&Open Plot File..."),
#if (KDE_VERSION_MAJOR > 2) || (KDE_VERSION_MINOR > 8)
                  "fileopen",
#else
                  QIconSet(BarIcon("fileopen")),
#endif
                  KStdAccel::open(), this, SLOT(slotFileOpenPlot()),
                  actionCollection(), "open_plot");
  a->setStatusText(i18n("Opens an existing plot file"));
  a = new KAction(i18n("Open &Data File..."),
#if (KDE_VERSION_MAJOR > 2) || (KDE_VERSION_MINOR > 8)
                  "fileopen",
#else
                  QIconSet(BarIcon("fileopen")),
#endif
                  0, this, SLOT(slotFileOpenData()), actionCollection(),
                  "open_data");
  a->setStatusText(i18n("Opens an existing data file"));
  recentFiles = KStdAction::openRecent(this,
                                       SLOT(openDocumentURL(const KURL&)),
                                       actionCollection());
  fileSave = KStdAction::save(this, SLOT(slotFileSave()), actionCollection());
  fileSave->setStatusText(i18n("Saves the current plot"));
  fileSaveAs = KStdAction::saveAs(this, SLOT(slotFileSaveAs()),
                                  actionCollection());
  fileSaveAs->setStatusText(i18n("Saves the plot under a different name"));
  fileClose = KStdAction::close(this, SLOT(slotFileClose()),
                                actionCollection());
  fileClose->setStatusText(i18n("Closes the current plot"));
  filePlot = new KAction(i18n("Display P&lot"), 0, this, SLOT(slotFilePlot()),
                         actionCollection(), "plot");
  filePlot->setStatusText(i18n("Displays the current plot"));
  filePSP = new KAction(i18n("&Portrait..."), 0, this, SLOT(slotFilePSP()),
                        actionCollection(), "postscript_p");
  filePSP->setStatusText(i18n("Generates PostScript file (portrait)"));
  filePSL = new KAction(i18n("&Landscape..."), 0, this, SLOT(slotFilePSL()),
                        actionCollection(), "postscript_l");
  filePSL->setStatusText(i18n("Generates PostScript file (landscape)"));
  filePrint = KStdAction::print(this, SLOT(slotFilePrint()),
                                actionCollection());
  filePrint->setStatusText(i18n("Prints the current plot"));
  a = new KAction(i18n("New &Window"), QIconSet(kapp->miniIcon()), 0, this,
                  SLOT(slotFileNewWindow()), actionCollection(), "new_window");
  a->setStatusText(i18n("Opens a new application window"));
  a = new KAction(i18n("Close W&indow"), 0, this, SLOT(slotFileCloseWindow()),
                  actionCollection(), "close_window");
  a->setStatusText(i18n("Closes the current window"));
  a = KStdAction::quit(this, SLOT(slotFileQuit()), actionCollection());
  a->setStatusText(i18n("Exits the program"));
  editUndo = KStdAction::undo(this, SLOT(slotEditUndo()), actionCollection());
  editUndo->setStatusText(i18n("Undoes the last change"));
  editRedo = KStdAction::redo(this, SLOT(slotEditRedo()), actionCollection());
  editRedo->setStatusText(i18n("Redoes the last change"));
  a = new KAction(i18n("&Items..."), 0, this, SLOT(slotEditItems()),
                  actionCollection(), "items");
  a->setStatusText(i18n("Edits plot items"));
  a = KStdAction::zoomIn(this, SLOT(slotViewZoomIn()), actionCollection());
  a->setStatusText(i18n("Increases magnification by 20 %"));
  a = KStdAction::zoomOut(this, SLOT(slotViewZoomOut()), actionCollection());
  a->setStatusText(i18n("Decreases magnification by 20 %"));
  a = KStdAction::zoom(this, SLOT(slotViewZoom()), actionCollection());
  a->setStatusText(i18n("Sets magnification"));
  viewReload = KStdAction::redisplay(this, SLOT(slotViewReload()),
                                     actionCollection());
  viewReload->setAccel(KStdAccel::reload());
  viewReload->setStatusText(i18n("Reloads file"));
  a = new KAction(i18n("Reload &Periodically..."), 0, this,
                  SLOT(slotViewAutoReload()), actionCollection(),
                  "autoreload");
  a->setStatusText(i18n("Reloads file periodically"));
  showToolbar = KStdAction::showToolbar(this, SLOT(slotOptionsToolBar()),
                                        actionCollection());
  showToolbar->setStatusText(i18n("Enables / disables the toolbar"));
  showStatusbar = KStdAction::showStatusbar(this, SLOT(slotOptionsStatusBar()),
                                            actionCollection());
  showStatusbar->setStatusText(i18n("Enables / disables the statusbar"));
  autoplot = new KToggleAction(i18n("Aut&oplot"), 0, this,
                               SLOT(slotOptionsAutoPlot()),
                               actionCollection(), "autoplot");
  autoplot->setStatusText(i18n("Enables / disables automatic plotting"));
  addData = new KToggleAction(i18n("Add &Files"), Key_Insert, this,
                              SLOT(slotOptionsAddData()),
                              actionCollection(), "add_data");
  addData->setStatusText(i18n("Enables / disables adding of data and plot files"));
  showSource = new KToggleAction(i18n("S&how Function Source"), 0, this,
                                 SLOT(slotOptionsShowSource()),
                                 actionCollection(), "show_source");
  showSource->setStatusText(i18n("Enables / disables display of "
                                 "function source"));
  boundingBox = new KToggleAction(i18n("Calculate &PS Bounding Box"), 0,
                                 actionCollection(), "bounding_box");
  boundingBox->setStatusText(i18n("Enables / disables calculation of "
                                  "PS bounding box"));
  saveAbsPath = new KToggleAction(i18n("Save Absol&ute Paths"), 0,
                              actionCollection(), "save_abs_path");
  saveAbsPath->setStatusText(i18n("Saves absolute paths of data "
                                  "and library files in plot files"));
  warning = new KToggleAction(i18n("Unsaved Changes &Warning"), 0,
                              actionCollection(), "warning");
  warning->setStatusText(i18n("Enables / disables warning on "
                              "unsaved changes"));
  saveAtEnd = new KToggleAction(i18n("Save Settings At &End"), 0,
                                actionCollection(), "save_at_end");
  saveAtEnd->setStatusText(i18n("Enables / disables saving settings at end"));
  a = KStdAction::keyBindings(this, SLOT(slotOptionsKeys()),
                              actionCollection());
  a->setStatusText(i18n("Displays key bindings dialog"));
  a = KStdAction::configureToolbars(this, SLOT(slotEditToolbars()),
                                    actionCollection());
  a->setStatusText(i18n("Displays toolbar configuration dialog"));
  a = new KAction(i18n("Configure Bas&ics..."), 0,
                  this, SLOT(slotOptionsBasic()),
                  actionCollection(), "configure_basic");
  a->setStatusText(i18n("Displays basic settings dialog"));
  a = new KAction(i18n("&Configure Autoplot..."), 0, this,
                  SLOT(slotOptionsAuto()),
                  actionCollection(), "configure_auto");
  a->setStatusText(i18n("Displays autoplot settings dialog"));
  a = new KAction(i18n("Co&nfigure Autofit..."), 0, this,
                  SLOT(slotOptionsAutoFit()),
                  actionCollection(), "configure_autofit");
  a->setStatusText(i18n("Displays autofit settings dialog"));
  a = KStdAction::saveOptions(this, SLOT(slotOptionsSave()),
                              actionCollection());
  a->setStatusText(i18n("Saves settings"));

#if KDE_VERSION == 303
  KAboutData* aboutData = new KAboutData("kpl", I18N_NOOP("Kpl"), VERSION,
                       "An application for plotting of data sets and functions",
                       KAboutData::License_GPL, "(c) 2002, Werner Stille",
                       0, 0, "stille@uni-freiburg.de");
  aboutData->addAuthor("Werner Stille", 0, "stille@uni-freiburg.de");
  KHelpMenu* hm = new KHelpMenu(this, aboutData);
  (KStdAction::helpContents(this, SLOT(slotHelpContents()), actionCollection(),
                           "help_contents"))->setShortcut(Key_F1);
  KStdAction::reportBug(hm, SLOT(reportBug()), actionCollection(),
                        "help_report_bug");
  KStdAction::aboutApp(hm, SLOT(aboutApplication()), actionCollection(),
                       "help_about_app");
  KStdAction::aboutKDE(hm, SLOT(aboutKDE()), actionCollection(),
                       "help_about_kde");
  setHelpMenuEnabled(false);
#endif
}

void KplApp::slotHelpContents()
{
#if KDE_VERSION == 303
  kapp->startServiceByDesktopName("konqueror", QString("help:/kpl/index.html"),
                                  0, 0, 0, "", true);
#endif
}

void KplApp::calcStatusBar()
{
  xPrec = QMIN(QMAX(2, Utils::int9(log10(QMAX(fabs(view->xmin),
                                              fabs(view->xmax)) / view->fxn
                                   + 1.0e-37) + 0.999999) +
               Utils::int9(log10(view->xscal * view->fxn + 1.0e-300) +
                           0.999999)), 16);
  yPrec = QMIN(QMAX(2, Utils::int9(log10(QMAX(fabs(view->ymin),
                                              fabs(view->ymax)) / view->fyn
                                   + 1.0e-37) + 0.999999) +
               Utils::int9(log10(-view->yscal * view->fyn + 1.0e-300) +
                           0.999999)), 16);
}

void KplApp::initStatusBar()
{
  statusBar()->insertItem("", StatusAdd, 0, true);
  statusBar()->insertFixedItem(" 10.00 cm,  15.00 cm ", StatusPos, true);
  statusBar()->insertFixedItem("  x = " +
                               QString::number(-10.0e-300 / 3.0, 'g', xPrec) +
                               ",  y = " +
                               QString::number(-10.0e-300 / 3.0, 'g', yPrec),
                               StatusVal, true);
  statusBar()->insertItem(i18n("Ready."), StatusMsg);
  statusBar()->changeItem("", StatusVal);
  statusBar()->changeItem("", StatusPos);
  slotOptionsAddData();
}

void KplApp::newStatusBar()
{
  int xold = xPrec;
  int yold = yPrec;
  calcStatusBar();
  if ((xPrec != xold) || (yPrec != yold)) {
    statusBar()->changeItem("  x = " +
                            QString::number(-10.0e-300 / 3.0, 'g', xPrec) +
                            ",  y = " +
                            QString::number(-10.0e-300 / 3.0, 'g', yPrec),
                            StatusVal);
    statusBar()->setItemFixed(StatusVal);
    statusBar()->changeItem("", StatusVal);
  }
}

void KplApp::initKeyAccel()
{
  key_accel = new KAccel(this);
  (new KAction(i18n("Scroll Left"), Key_Left, view, SLOT(slotScrollLeft()),
               actionCollection(), "cursor_left"))->plugAccel(key_accel);
  (new KAction(i18n("Scroll Right"), Key_Right, view, SLOT(slotScrollRight()),
               actionCollection(), "cursor_right"))->plugAccel(key_accel);
  (new KAction(i18n("Scroll Up"), Key_Up, view, SLOT(slotScrollUp()),
               actionCollection(), "cursor_up"))->plugAccel(key_accel);
  (new KAction(i18n("Scroll Down"), Key_Down, view, SLOT(slotScrollDown()),
               actionCollection(), "cursor_down"))->plugAccel(key_accel);
  (new KAction(i18n("Scroll Page Up"), Key_PageUp, view,
               SLOT(slotScrollPageUp()), actionCollection(),
               "page_up"))->plugAccel(key_accel);
  (new KAction(i18n("Scroll Page Down"), Key_PageDown, view,
               SLOT(slotScrollPageDown()), actionCollection(),
               "page_down"))->plugAccel(key_accel);
  (new KAction(i18n("Scroll Page Left"), CTRL + Key_PageUp, view,
               SLOT(slotScrollPageLeft()), actionCollection(),
               "page_left"))->plugAccel(key_accel);
  (new KAction(i18n("Scroll Page Right"), CTRL + Key_PageDown, view,
               SLOT(slotScrollPageRight()), actionCollection(),
               "page_right"))->plugAccel(key_accel);
  (new KAction(i18n("Scroll to Left Margin"), CTRL + Key_Home, view,
               SLOT(slotScrollHomeLeft()), actionCollection(),
               "margin_left"))->plugAccel(key_accel);
  (new KAction(i18n("Scroll to Bottom Margin"), Key_Home, view,
               SLOT(slotScrollHomeDown()), actionCollection(),
               "margin_bottom"))->plugAccel(key_accel);
  (new KAction(i18n("Scroll to Right Margin"), CTRL + Key_End, view,
               SLOT(slotScrollEndRight()), actionCollection(),
               "margin_right"))->plugAccel(key_accel);
  (new KAction(i18n("Scroll to Top Margin"), Key_End, view,
               SLOT(slotScrollEndUp()), actionCollection(),
               "margin_top"))->plugAccel(key_accel);
  createGUI();
  if (!showToolbar->isChecked())
    toolBar("mainToolBar")->hide();
  toolBar("mainToolBar")->setBarPos(toolBarPos);
}

void KplApp::initDocument()
{
  doc = new KplDoc(this);
  doc->newDocument();
}

void KplApp::initView()
{
  view = new KplView(this, fz);
  setCentralWidget(view);
}

void KplApp::saveOptions(bool saveAll)
{
  recentFiles->saveEntries(config);
  if (itemDlg)
    Utils::saveGeometry(itemDlg, "ItemDialog");
  config->setGroup(kapp->instanceName());
  if (saveAll) {
    config->writeEntry("ShowToolbar", showToolbar->isChecked());
    config->writeEntry("ShowStatusbar", showStatusbar->isChecked());
    config->writeEntry("ShowItemDialog", itemDlg ?
                       itemDlg->isVisible() : false);
    config->writeEntry("ToolBarPos", (int) toolBar("mainToolBar")->barPos());
    config->writeEntry("Directory", doc->currentDir());
    config->writeEntry("Width", width());
    config->writeEntry("Height", height());
    config->writeEntry("AutoPlot", autoplot->isChecked());
    config->writeEntry("AddData", addData->isChecked());
    config->writeEntry("ShowSource", showSource->isChecked());
    config->writeEntry("BoundingBox", boundingBox->isChecked());
    config->writeEntry("SaveAbsPath", saveAbsPath->isChecked());
    config->writeEntry("Warning", warning->isChecked());
    config->writeEntry("SaveAtEnd", saveAtEnd->isChecked());
    config->writeEntry("AutoIndexX", doc->aut.ixAuto);
    config->writeEntry("AutoIndexY", doc->aut.iyAuto);
    config->writeEntry("AutoIndexE", doc->aut.ieAuto);
    config->writeEntry("AutoNorm", doc->aut.autoNorm);
    config->writeEntry("AutoErrBar", doc->aut.autoErr);
    config->writeEntry("AutoLogX", doc->aut.autoLogX);
    config->writeEntry("AutoLogY", doc->aut.autoLogY);
    config->writeEntry("AutoGrid", doc->aut.autoGrid);
    config->writeEntry("AutoPathHeader", doc->aut.autohPath);
    config->writeEntry("AutoFit", doc->aut.autoFit);
    config->writeEntry("AutoSymbol", doc->aut.autoSymb);
    config->writeEntry("AutoX0", doc->aut.x0Auto);
    config->writeEntry("AutoY0", doc->aut.y0Auto);
    config->writeEntry("AutoXl", doc->aut.xlAuto);
    config->writeEntry("AutoYl", doc->aut.ylAuto);
    config->writeEntry("FitErrorColumn", doc->aut.err.fitErrCol);
    config->writeEntry("FitNonLinear", doc->aut.fitNonLin);
    config->writeEntry("FitShowDialog", doc->aut.fitShowDlg);
    config->writeEntry("FitSaveParam", doc->aut.fitSavePar);
    config->writeEntry("FitFollow", doc->aut.fitFollow);
    config->writeEntry("FitMaxIterations", doc->aut.fitMaxIt);
    config->writeEntry("FitTolerance", doc->aut.fitTol);
    config->writeEntry("FitSymbol", doc->aut.fitSymb);
    config->writeEntry("FitColor", KplGraph::rgbQt1(doc->aut.fitColor));
    config->writeEntry("FitPath", doc->aut.fitPath);
    config->writeEntry("FitName", doc->aut.fitName);
    config->writeEntry("ErrorModelPath", doc->aut.err.errModPath);
    config->writeEntry("ErrorModelName", doc->aut.err.errModName);
    QStrList s1, s2, s3;
    for (int i = 0; i < KPL_NPMAX; i++) {
      s1.insert(i, QString::number((int) doc->aut.fitPar[i]));
      s2.insert(i, doc->number(doc->aut.pFit[i]));
      s3.insert(i, doc->number(doc->aut.err.pErrMod[i]));
    }
    config->writeEntry("FitEnableParam", s1, ' ');
    config->writeEntry("FitParam", s2, ' ');
    config->writeEntry("ErrorModelParam", s3, ' ');
    config->writeEntry("FormatX", doc->aut.xf);
    config->writeEntry("FormatY", doc->aut.yf);
    config->writeEntry("ColorFrame", KplGraph::rgbQt1(doc->aut.colFrame));
    config->writeEntry("ColorGrid", KplGraph::rgbQt1(doc->aut.colGrid));
    config->writeEntry("ColorData", KplGraph::rgbQt1(doc->aut.colData));
    config->writeEntry("OutputFormat", QString(QChar(doc->aut.format)));
    config->writeEntry("Precision", doc->aut.prec);
    config->writeEntry("Separator", doc->aut.iSep);
    config->writeEntry("Zoom", view->zoom());
    config->writeEntry("AutoLetX", doc->aut.sxAuto);
    config->writeEntry("AutoLetY", doc->aut.syAuto);
    config->writeEntry("AutoLetH", doc->aut.shAuto);
  }
  config->sync();
}

void KplApp::readOptions()
{
  config->setGroup(kapp->instanceName());
  config->setDollarExpansion();
  resize(config->readNumEntry("Width", 640),
         config->readNumEntry("Height", 512));
  recentFiles->loadEntries(config);
  showToolbar->setChecked(config->readBoolEntry("ShowToolbar", true));
  showStatusbar->setChecked(config->readBoolEntry("ShowStatusbar", true));
  if (!showStatusbar->isChecked())
    statusBar()->hide();
  if (config->readBoolEntry("ShowItemDialog")) {
    slotEditItems();
    config->setGroup(kapp->instanceName());
  }
  toolBarPos = (KToolBar::BarPosition) config->readNumEntry("ToolBarPos",
                                                            KToolBar::Top);
  QString kplDir = KGlobal::dirs()->findResourceDir("appdata", "demo1.plo");
  doc->setCurrentDir(config->readEntry("Directory", kplDir) + "x");
  doc->aut.showSource = config->readBoolEntry("ShowSource");
  showSource->setChecked(doc->aut.showSource);
  addData->setChecked(config->readBoolEntry("AddData"));
  autoplot->setChecked(config->readBoolEntry("AutoPlot", true));
  boundingBox->setChecked(config->readBoolEntry("BoundingBox"));
  saveAbsPath->setChecked(config->readBoolEntry("SaveAbsPath", true));
  warning->setChecked(config->readBoolEntry("Warning", true));
  saveAtEnd->setChecked(config->readBoolEntry("SaveAtEnd"));
  doc->aut.ixAuto = config->readNumEntry("AutoIndexX");
  doc->aut.iyAuto = config->readNumEntry("AutoIndexY", 1);
  doc->aut.ieAuto = config->readNumEntry("AutoIndexE", 2);
  doc->aut.autoNorm = config->readBoolEntry("AutoNorm", true);
  doc->aut.autoErr = config->readBoolEntry("AutoErrBar");
  doc->aut.autoLogX = config->readBoolEntry("AutoLogX");
  doc->aut.autoLogY = config->readBoolEntry("AutoLogY");
  doc->aut.autoGrid = config->readNumEntry("AutoGrid",
                                           KplGraph::AxesWithLabels);
  doc->aut.autohPath = config->readBoolEntry("AutoPathHeader", true);
  doc->aut.autoFit = config->readBoolEntry("AutoFit");
  doc->aut.autoSymb = config->readNumEntry("AutoSymbol", 1);
  doc->aut.x0Auto = config->readDoubleNumEntry("AutoX0", 4.0);
  doc->aut.y0Auto = config->readDoubleNumEntry("AutoY0", 3.0);
  doc->aut.xlAuto = config->readDoubleNumEntry("AutoXl", 15.0);
  doc->aut.ylAuto = config->readDoubleNumEntry("AutoYl", 10.0);
  doc->aut.err.fitErrCol = config->readBoolEntry("FitErrorColumn");
  doc->aut.fitNonLin = config->readBoolEntry("FitNonLinear", true);
  doc->aut.fitShowDlg = config->readBoolEntry("FitShowDialog");
  doc->aut.fitSavePar = config->readBoolEntry("FitSaveParam");
  doc->aut.fitFollow = config->readBoolEntry("FitFollow");
  doc->aut.fitMaxIt = config->readNumEntry("FitMaxIterations", 100);
  doc->aut.fitTol = config->readDoubleNumEntry("FitTolerance", 1e-14);
  doc->aut.fitSymb = config->readNumEntry("FitSymbol", 1);
  doc->aut.fitColor = KplGraph::rgbQt(config->readUnsignedNumEntry("FitColor"));
  doc->aut.fitPath = config->readEntry("FitPath", kplDir + "fkt.so");
  doc->aut.fitName = config->readEntry("FitName", "polynom");
  doc->aut.err.errModPath = config->readEntry("ErrorModelPath",
                                              kplDir + "fkt.so");
  doc->aut.err.errModName = config->readEntry("ErrorModelName", "polynom");
  QStringList list = config->readListEntry("FitEnableParam", ' ');
  int cnt = list.count();
  for (int i = 0; i < KPL_NPMAX; i++)
    doc->aut.fitPar[i] = (i < cnt) ? list[i].toInt() : false;
  list = config->readListEntry("FitParam", ' ');
  cnt = list.count();
  for (int i = 0; i < KPL_NPMAX; i++)
    doc->aut.pFit[i] = (i < cnt) ? list[i].toDouble() : 0.0;
  list = config->readListEntry("ErrorModelParam", ' ');
  cnt = list.count();
  for (int i = 0; i < KPL_NPMAX; i++)
    doc->aut.err.pErrMod[i] = (i < cnt) ? list[i].toDouble() : 0.0;
  if (!cnt)
    doc->aut.err.pErrMod[0] = 0.01;
  doc->aut.xf = config->readDoubleNumEntry("FormatX", 21.0);
  doc->aut.yf = config->readDoubleNumEntry("FormatY", 29.7);
  doc->aut.colFrame = KplGraph::rgbQt(config->readUnsignedNumEntry("ColorFrame"));
  doc->aut.colGrid = KplGraph::rgbQt(config->readUnsignedNumEntry("ColorGrid"));
  doc->aut.colData = KplGraph::rgbQt(config->readUnsignedNumEntry("ColorData"));
  doc->aut.format = *(config->readEntry("OutputFormat", "g").latin1());
  doc->aut.prec = config->readNumEntry("Precision", 6);
  doc->aut.iSep = config->readNumEntry("Separator");
  fz = config->readDoubleNumEntry("Zoom", 1.0);
  config->setDollarExpansion(false);
  doc->aut.sxAuto = config->readEntry("AutoLetX", "x");
  doc->aut.syAuto = config->readEntry("AutoLetY", "y");
  doc->aut.shAuto = config->readEntry("AutoLetH", "");
 }

void KplApp::openRead(int type)
{
  KURL::List list = KFileDialog::getOpenURLs(doc->currentDir(),
                                             type == KplDoc::Data ?
                                             "*.dat *.DAT\n*" :
                                             "*.plo *.PLO\n*");
  if (!list.isEmpty())
    openList(&list, type);
}

bool KplApp::save(bool current)
{
  bool success = false;
  if ((doc->URL().fileName() != i18n("Untitled")) && current &&
      (doc->fType == KplDoc::Plot)) {
    success = doc->saveDocument(doc->URL(), doc->items(),
                                saveAbsPath->isChecked());
    if (success) {
      doc->setModified(false);
      doc->fType = KplDoc::Plot;
      updWin();
    }
  } else {
    KURL url;
    if (Utils::getWriteURL(this, url, "*.plo\n*", doc))
      success = saveAs(url);
  }
  return success;
}

bool KplApp::saveAs(const KURL& url)
{
  bool success = doc->saveDocument(url, doc->items(),
                                   saveAbsPath->isChecked());
  if (success) {
    doc->setURL(url);
    doc->setModified(false);
    doc->fType = KplDoc::Plot;
    slotUpdate(false, true);
  }
  return success;
}

bool KplApp::queryClose()
{
  if (doc->isModified())
    do {
      int i = KMessageBox::warningYesNoCancel(this,
        i18n("The current plot has been changed.\nSave file?"));
      if (i == KMessageBox::Cancel)
        return false;
      if (i == KMessageBox::No)
        break;
    } while (!save(true));
  return true;
}

bool KplApp::queryExit()
{
  saveOptions(saveAtEnd->isChecked());
  return true;
}

void KplApp::saveProperties(KConfig* _cfg)
{
  bool untitled = (doc->URL().fileName() == i18n("Untitled"));
  if (untitled && !doc->isModified())
    return;
  QString fileName;
  if (untitled)
    fileName = QDir::homeDirPath() + "/" + i18n("Untitled");
  else
    fileName = doc->URL().url();
  _cfg->writeEntry("filename", fileName);
  _cfg->writeEntry("modified", doc->isModified());
  if (doc->isModified())
    doc->saveDocument(kapp->tempSaveName(fileName), doc->items());
}

void KplApp::readProperties(KConfig* _cfg)
{
  QString filename = _cfg->readEntry("filename", "");
  KURL u = KURL(filename);
  bool recovered = false;
  if (_cfg->readBoolEntry("modified")) {
    bool canRecover;
    QString tempname = kapp->checkRecoverFile(filename, canRecover);
    if (canRecover) {
      doc->openDocument(tempname, KplDoc::Plot);
      QFile::remove(tempname);
      doc->setModified();
      recovered = true;
    }
  } else
    if (!filename.isEmpty()) {
      doc->openDocument(u);
      recovered = true;
    }
  if (recovered)
    doc->setURL(u);
  updWin();
}

bool KplApp::savePSURL(const KURL& url, bool landscape)
{
  QString filename = url.isLocalFile() ? url.path() : doc->tmpFile();
  QFile f(filename);
  bool success = f.open(IO_WriteOnly);
  if (success) {
    view->QScrollView::viewport()->setCursor(waitCursor);
    PSGraph g;
    QTextStream ts(&f);
    g.psInit(&ts, landscape ? doc->aut.yf : doc->aut.xf,
             landscape ? doc->aut.xf : doc->aut.yf);
    if (landscape)
      ts << QString::number(qRound(doc->aut.yf * PSGraph::pts))
         << " 0 translate 90 rotate\n";
    for (KplItem* item = doc->items()->first(); item;
         item = doc->items()->next())
      item->draw(&g);
    g.psExit();
    f.close();
    if (boundingBox->isChecked()) {
      QString tmpPBM = QDir::homeDirPath() + "/kpltmp.pbm";
      QCString s2;
      KProcess kp;
      kp << "gs" << "-dNOPAUSE" << "-dBATCH" << "-q" << "-sDEVICE=pbmraw"
         << ("-sOutputFile=" + tmpPBM) << "-r72"
         << s2.sprintf("-g%ix%i", qRound(PSGraph::pts *
           (landscape ? doc->aut.yf : doc->aut.xf)),
           qRound(PSGraph::pts * (landscape ? doc->aut.xf : doc->aut.yf)))
         << filename;
      kp.start(KProcess::Block);
      QImage image;
      if (image.load(tmpPBM, "PBMRAW")) {
        int x, y, xl = 0, yo = 0;
        int xr = image.width();
        int yu = image.height();
        for (y = 0; y < image.height(); y++)
          for (x = 0; x < image.width(); x++)
            if (!(image.pixel(x, y) & 0xffffff)) {
              yo = y;
              xl = xr = x;
              goto yo_found;
            }
yo_found:
        for (y = image.height() - 1; y > yo; y--)
          for (x = 0; x < image.width(); x++)
            if (!(image.pixel(x, y) & 0xffffff)) {
              yu = y;
              xl = QMIN(xl, x);
              xr = QMAX(xr, x);
              goto yu_found;
            }
yu_found:
        for (y = yo; y <= yu; y++){
          for (x = 0; x < xl; x++)
            if (!(image.pixel(x, y) & 0xffffff)) {
              xl = x;
              break;
            }
          for (x = image.width() - 1; x > xr; x--)
            if (!(image.pixel(x, y) & 0xffffff)) {
              xr = x;
              break;
            }
        }
        xl = QMAX(xl - 2, 0);
        xr = QMIN(xr + 2, image.width() - 1);
        yu = QMAX(image.height() - yu - 2, 0);
        yo = QMIN(image.height() - yo + 2, image.height() - 1);
        remove(tmpPBM);
        f.open(IO_ReadWrite);
        ts << "%!PS-Adobe-3.0 EPSF-3.0\n%%BoundingBox: "
           << QString::number(xl) << " " << QString::number(yu) << " "
           << QString::number(xr) << " " << QString::number(yo) << "      ";
        f.close();
      } else
        KMessageBox::error(this,
          i18n("Bitmap file for bounding box calculation not loaded!"));
    }
    doc->setCurrentDir(url);
    if (!url.isLocalFile())
      doc->copyTmp(url);
    view->QScrollView::viewport()->setCursor(crossCursor);
  } else
    KMessageBox::sorry(this, i18n("Writing to this file not possible."));
  return success;
}

void KplApp::slotFileNew()
{
  if (queryClose())
    newPlot();
  slotStatusMsg(i18n("Ready."));
}

void KplApp::slotFileOpenPlot()
{
  statusBar()->message(i18n("Opening file..."));
  openRead(KplDoc::Plot);
  slotStatusMsg(i18n("Ready."));
}

void KplApp::slotFileOpenData()
{
  statusBar()->message(i18n("Opening file..."));
  openRead(KplDoc::Data);
  slotStatusMsg(i18n("Ready."));
}

void KplApp::slotFileSave()
{
  statusBar()->message(i18n("Saving file..."));
  save(true);
  slotStatusMsg(i18n("Ready."));
}

void KplApp::slotFileSaveAs()
{
  statusBar()->message(i18n("Saving file under different name..."));
  save(false);
  slotStatusMsg(i18n("Ready."));
}

void KplApp::slotFileClose()
{
  statusBar()->message(i18n("Closing file..."));
  if (queryClose()) {
    doc->newDocument();
    doc->backupItems();
  }
  slotStatusMsg(i18n("Ready."));
}

void KplApp::slotFilePlot()
{
  statusBar()->message(i18n("Plotting..."));
  plot = doc->items()->count();
  slotPlotCond();
  slotStatusMsg(i18n("Ready."));
}

void KplApp::slotPlotCond()
{
  if (plot)
    view->paintWidget();
  newStatusBar();
}

void KplApp::slotFilePrint()
{
  statusBar()->message(i18n("Printing..."));
#if (KDE_VERSION_MAJOR > 2) || (KDE_VERSION_MINOR > 1)
  KPrinter printer;
#else
  QPrinter printer;
#endif
  if (printer.setup(this))
    view->print(&printer);
  slotStatusMsg(i18n("Ready."));
}

void KplApp::slotFilePSP()
{
  slotFilePS(false);
}

void KplApp::slotFilePSL()
{
  slotFilePS(true);
}

void KplApp::slotFilePS(bool landscape)
{
  statusBar()->message(i18n("Generating PostScript file..."));
  KURL u;
  if (Utils::getWriteURL(this, u, "*.ps *.eps\n*", doc))
    if (!savePSURL(u, landscape))
      KMessageBox::sorry(this, i18n("Writing to this file not possible."));
  slotStatusMsg(i18n("Ready."));
}

void KplApp::slotFileNewWindow()
{
  (new KplApp)->show();
  slotStatusMsg(i18n("Ready."));
}

void KplApp::slotFileCloseWindow()
{
  close();
}

void KplApp::slotFileQuit()
{
  slotStatusMsg(i18n("Exiting..."));
  if (memberList)
    for (KMainWindow* w = memberList->first(); w; w = memberList->first())
      if (!w->close()) {
        slotStatusMsg(i18n("Ready."));
        break;
      }
}

void KplApp::slotEditUndo()
{
  doc->restoreItems();
}

void KplApp::slotEditRedo()
{
  doc->restoreItems(false);
}

void KplApp::slotEditItems()
{
  if (!itemDlg) {
    itemDlg = new ItemDlg(doc);
    connect(itemDlg, SIGNAL(displayMessage(const QString&)),
            SLOT(slotStatusMsg(const QString&)));
  }
  itemDlg->show();
  itemDlg->raise();
}

void KplApp::slotViewZoomIn()
{
  view->setZoom(0.01 * (QMIN(qRound(100 * view->zoom()) + 20, 2000)));
  shrink();
  slotStatusMsg(i18n("Ready."));
}

void KplApp::slotViewZoomOut()
{
  view->setZoom(0.01 * (QMAX(qRound(100 * view->zoom()) - 20, 1)));
  shrink();
  slotStatusMsg(i18n("Ready."));
}

void KplApp::slotViewZoom()
{
  statusBar()->message(i18n("Setting magnification..."));
  int i = qRound(100 * view->zoom());
  ZoomDlg dlg(this, &i);
  if (dlg.exec()) {
    view->setZoom(0.01 * i);
    shrink();
  }
  slotStatusMsg(i18n("Ready."));
}

void KplApp::slotWatchFile()
{
  if (doc->URL().isLocalFile()) {
    QFileInfo fi(doc->URL().path());
    if (fi.lastModified() == doc->URLTime())
      return;
  }
  slotViewReload();
}

void KplApp::slotViewReload()
{
  statusBar()->message(i18n("Reloading file..."));
  autoplot->setChecked(true);
  bool sav = doc->aut.addData;
  doc->aut.addData = false;
  doc->openDocument(KURL(doc->URL()), doc->fType);
  doc->aut.addData = sav;
  slotStatusMsg(i18n("Ready."));
}

void KplApp::slotViewAutoReload()
{
  statusBar()->message(i18n("Reloading file periodically..."));
  bool act = timer->isActive();
  int tRef = tRefresh;
  RefreshDlg dlg(this, &act, &tRef);
  if (dlg.exec()) {
    if (timer->isActive() && ((tRef != tRefresh) || (act != timer->isActive())))
      timer->stop();
    if (act && ((tRef != tRefresh) || (act != timer->isActive()))) {
      timer->start(1000 * tRef);
    }
    tRefresh = tRef;
  }
  slotStatusMsg(i18n("Ready."));
}

void KplApp::slotOptionsToolBar()
{
  toolBar("mainToolBar")->enable(KToolBar::Toggle);
}

void KplApp::slotOptionsStatusBar()
{
  if (statusBar()->isVisible())
    statusBar()->hide();
  else
    statusBar()->show();
}

void KplApp::slotOptionsAutoPlot()
{
  if (autoplot->isChecked())
    slotFilePlot();
}

void KplApp::slotOptionsAddData()
{
  doc->aut.addData = addData->isChecked();
  statusBar()->changeItem(i18n(doc->aut.addData ? "Add" : "Replace"),
                          StatusAdd);
}

void KplApp::slotOptionsShowSource()
{
  doc->aut.showSource = showSource->isChecked();
}

void KplApp::slotOptionsKeys()
{
  statusBar()->message(i18n("Key bindings..."));
  KKeyDialog::configureKeys(actionCollection(), xmlFile());
  slotStatusMsg(i18n("Ready."));
}

void KplApp::slotEditToolbars()
{
  statusBar()->message(i18n("Toolbar configuration..."));
  KEditToolbar dlg(actionCollection());
  if (dlg.exec()) {
    toolBarPos = toolBar("mainToolBar")->barPos();
    createGUI();
    toolBar("mainToolBar")->setBarPos(toolBarPos);
  }
  slotStatusMsg(i18n("Ready."));
}

void KplApp::slotOptionsBasic()
{
  statusBar()->message(i18n("Basic settings..."));
  BasicDlg dlg(this, &doc->aut);
  if (dlg.exec())
    shrink();
  slotStatusMsg(i18n("Ready."));
}

void KplApp::slotOptionsAuto()
{
  statusBar()->message(i18n("Autoplot settings..."));
  AutoDlg dlg(this, doc);
  dlg.exec();
  slotStatusMsg(i18n("Ready."));
}

void KplApp::slotOptionsAutoFit()
{
  statusBar()->message(i18n("Autofit settings..."));
  AutoFitDlg dlg(this, doc);
  dlg.exec();
  slotStatusMsg(i18n("Ready."));
}

void KplApp::slotOptionsSave()
{
  saveOptions(true);
  slotStatusMsg(i18n("Ready."));
}

void KplApp::slotStatusMsg(const QString &text)
{
  statusBar()->clear();
  statusBar()->changeItem(text, StatusMsg);
}

void KplApp::slotStatusHelpMsg(const QString &text)
{
  statusBar()->message(text, 3000);
}

void KplApp::slotUpdate(bool updPlot, bool updList)
{
  updWin();
  if (updPlot)
    if (doc->items()->count())
      slotPlotCond();
    else
      view->eraseWidget();
  if (itemDlg && updList)
    itemDlg->updItemList();
}

void KplApp::updWin()
{
  if (doc->URL().fileName() == i18n("Untitled"))
    setCaption(doc->URL().fileName(), doc->isModified());
  else {
    recentFiles->addURL(doc->URL());
    if (doc->URL().isLocalFile())
      setCaption(doc->URL().path(), doc->isModified());
    else
      setCaption(doc->URL().url(), doc->isModified());
  }
  recentFiles->setEnabled(recentFiles->items().count());
  bool nItems = doc->items()->count();
  fileSaveAs->setEnabled(nItems);
  fileClose->setEnabled(nItems);
  filePlot->setEnabled(nItems);
  filePSP->setEnabled(nItems);
  filePSL->setEnabled(nItems);
  filePrint->setEnabled(nItems);
  fileSave->setEnabled(doc->isModified());
  editUndo->setEnabled(doc->undoAllowed());
  editRedo->setEnabled(doc->redoAllowed());
  viewReload->setEnabled((doc->URL().fileName() != i18n("Untitled")));
  plot = autoplot->isChecked() && nItems;
}

void KplApp::updStatus(const QPoint& p, bool showValues)
{
  if (statusBar()->isVisible()) {
    double xcm, ycm, xr, yr;
    if ((!view->i2cm(p, &xcm, &ycm)) && plot && showValues) {
      view->i2r(p, view->fxn, view->fyn, &xr, &yr);
      statusBar()->changeItem("  x = " + QString::number(xr, 'g', xPrec) +
                              ",  y = " + QString::number(yr, 'g', yPrec),
                              StatusVal);
    } else
      statusBar()->changeItem("", StatusVal);
    QCString s;
    if ((xcm < doc->aut.xf) && (ycm >= 0.0))
      s.sprintf("%6.2f cm, %6.2f cm", xcm, ycm);
    statusBar()->changeItem(s, StatusPos);
  }
}

void KplApp::clearStatus()
{
  if (statusBar()->isVisible()) {
    statusBar()->changeItem("", StatusVal);
    statusBar()->changeItem("", StatusPos);
  }
}

void KplApp::shrink()
{
  view->layout();
  if ((QMAX(view->contentsWidth(), 120) < view->frameRect().width()) ||
      (view->contentsHeight() < view->frameRect().height())) {
    int w = QMIN(QMAX(view->contentsWidth(), 120), view->frameRect().width()) -
            view->frameRect().width() + width();
    resize(w, QMIN(view->contentsHeight(), view->frameRect().height()) -
           view->frameRect().height() + height() +
           menuBar()->heightForWidth(w) - menuBar()->height());
    newStatusBar();
  }
  view->eraseWidget();
  slotPlotCond();
}

void KplApp::showPopupMenu()
{
  ((QPopupMenu*) factory()->container("popup", this))->exec(QCursor::pos());
}

void KplApp::newPlot()
{
  doc->newDocument();
  doc->backupItems();
}

int KplApp::openPlot(const QString& url)
{
  return doc->openDocument(KURL(url), KplDoc::Plot) ? 1 : 0;
}

int KplApp::openData(const QString& url)
{
  return doc->openDocument(KURL(url), KplDoc::Data) ? 1 : 0;
}

void KplApp::doPlot()
{
  slotFilePlot();
}

int KplApp::savePlot(const QString& url)
{
  if (doc->items()->count())
    return saveAs(KURL(url)) ? 1 : 0;
  else
    return 0;
}

int KplApp::savePS(const QString& url, int landscape)
{
  if (doc->items()->count())
    return savePSURL(KURL(url), landscape) ? 1 : 0;
  else
    return 0;
}

int KplApp::printPlot(const QString& printerName, const QString& fileName,
                      int landscape, int numCopies)
{
  if (doc->items()->count()) {
#if (KDE_VERSION_MAJOR > 2) || (KDE_VERSION_MINOR > 1)
    KPrinter p;
#else
    QPrinter p;
#endif
    p.setPrinterName(printerName);
    p.setOutputFileName(fileName);
#if (KDE_VERSION_MAJOR > 2) || (KDE_VERSION_MINOR > 1)
    p.setOrientation(landscape ? KPrinter::Landscape : KPrinter::Portrait);
#else
    p.setOrientation(landscape ? QPrinter::Landscape : QPrinter::Portrait);
#endif
    p.setNumCopies(numCopies);
    view->print(&p);
    return 1;
  } else
    return 0;
}

void KplApp::setAutoplotOption(int setting)
{
  autoplot->setChecked(setting);
}

void KplApp::setAddFilesOption(int setting)
{
  addData->setChecked(setting);
  slotOptionsAddData();
}

void KplApp::setPathOption(int setting)
{
  saveAbsPath->setChecked(setting);
}

void KplApp::setBoundingBoxOption(int setting)
{
  boundingBox->setChecked(setting);
}

void KplApp::setPageFormat(double w, double h)
{
  doc->aut.xf = w;
  doc->aut.yf = h;
  shrink();
}

void KplApp::setColors(const QString& colFrame, const QString& colGrid,
                       const QString& colData)
{
  unsigned u;
  sscanf(colFrame.latin1(), "%x", &u);
  doc->aut.colFrame = KplGraph::rgbQt(u);
  sscanf(colGrid.latin1(), "%x", &u);
  doc->aut.colGrid = KplGraph::rgbQt(u);
  sscanf(colData.latin1(), "%x", &u);
  doc->aut.colData = KplGraph::rgbQt(u);
}

void KplApp::setOutputFormat(const QString& pres, int prec)
{
  doc->aut.format = pres[0].latin1();
  doc->aut.prec = prec;
}

void KplApp::setAutoplot(double x0, double w, double y0, double h,
                         int gridmode, const QString& sx, const QString& sy,
                         const QString& sh, int pathHeading, int ix, int iy,
                         int ie, int logx, int logy, int errbars, int symb,
                         int autoNorm, int autoFit)
{
  doc->aut.x0Auto = x0;
  doc->aut.xlAuto = w;
  doc->aut.y0Auto = y0;
  doc->aut.ylAuto = h;
  doc->aut.autoGrid = gridmode;
  doc->aut.sxAuto = sx;
  doc->aut.syAuto = sy;
  doc->aut.shAuto = sh;
  doc->aut.autohPath = pathHeading;
  doc->aut.ixAuto = ix;
  doc->aut.iyAuto = iy;
  doc->aut.ieAuto = ie;
  doc->aut.autoLogX = logx;
  doc->aut.autoLogY = logy;
  doc->aut.autoErr = errbars;
  doc->aut.autoSymb = symb;
  doc->aut.autoNorm = autoNorm;
  doc->aut.autoFit = autoFit;
}

void KplApp::setAutofit(const QString& path, const QString& _name,
                        int nonLin, int errCol,
                        const QString& errModPath, const QString& errModName,
                        int itMax, double tol, int symb, const QString& color,
                        int showDlg, int savePar, int follow)
{
  doc->aut.fitPath = path;
  doc->aut.fitName = _name;
  doc->aut.fitNonLin = nonLin;
  if (nonLin) {
    doc->aut.fitMaxIt = itMax;
    doc->aut.fitTol = tol;
  }
  doc->aut.err.fitErrCol = errCol;
  if (!errCol) {
    doc->aut.err.errModPath = errModPath;
    doc->aut.err.errModName = errModName;
  }
  doc->aut.fitSymb = symb;
  unsigned u;
  sscanf(color.latin1(), "%x", &u);
  doc->aut.fitColor = KplGraph::rgbQt(u);
  doc->aut.fitShowDlg = showDlg;
  doc->aut.fitSavePar = savePar;
  doc->aut.fitFollow = follow;
}

int KplApp::loadAutofitPar(const QString& url)
{
  return Utils::loadPar(KURL(url), this, doc->aut.pFit, doc) ? 1 : 0;
}

int KplApp::setAutofitPar(int iPar, double value)
{
  if ((iPar < 0) || (iPar >= KPL_NPMAX))
    return 0;
  doc->aut.pFit[iPar] = value;
  return 1;
}

int KplApp::enableFitPar(int iPar, int fit)
{
  if ((iPar < 0) || (iPar >= KPL_NPMAX))
    return 0;
  doc->aut.fitPar[iPar] = fit;
  return 1;
}

int KplApp::loadErrModPar(const QString& url, int iArr)
{
  int success = 0;
  if (iArr && (unsigned(iArr) >= arr.count()))
    return success;
  double p[KPL_NPMAX];
  success = Utils::loadPar(KURL(url), this, p, doc) ? 1 : 0;
  if (success) {
    QString s;
    s.sprintf("DataErrors%i", iArr);
    config->setGroup(s);
    QStrList s1;
    for (int i = 0; i < KPL_NPMAX; i++) {
      s1.insert(i, doc->number(p[i]));
      if (!iArr)
        doc->aut.err.pErrMod[i] = p[i];
    }
    config->writeEntry("ErrorModelParam", s1, ' ');
  }
  return success;
}

int KplApp::setErrModPar(int iPar, double value, int iArr)
{
  if ((iPar < 0) || (iPar >= KPL_NPMAX))
    return 0;
  if (iArr && (unsigned(iArr) >= arr.count()))
    return 0;
  QString s;
  s.sprintf("DataErrors%i", iArr);
  config->setGroup(s);
  QStringList list = config->readListEntry("ErrorModelParam", ' ');
  int cnt = list.count();
  double p[KPL_NPMAX];
  for (int i = 0; i < KPL_NPMAX; i++)
    p[i] = (i < cnt) ? list[i].toDouble() : 0.0;
  p[iPar] = value;
  QStrList s3;
  for (int i = 0; i < KPL_NPMAX; i++)
    s3.insert(i, doc->number(p[i]));
  config->writeEntry("ErrorModelParam", s3, ' ');
  if (!iArr)
    doc->aut.err.pErrMod[iPar] = value;
  return 1;
}

int KplApp::deleteItem(int iItem)
{
  if (unsigned(iItem) < doc->items()->count()) {
    doc->deleteItem(iItem);
    return 1;
  } else
    return 0;
}

int KplApp::moveItem(int is, int id)
{
  if ((unsigned(is) < doc->items()->count()) &&
      (unsigned(id) < doc->items()->count()) && (is != id)) {
    doc->moveItem(is, id);
    return 1;
  } else
    return 0;
}

int KplApp::activateItem(int iItem, int active)
{
  if (unsigned(iItem) < doc->items()->count()) {
    doc->items()->at(iItem)->setActive(active);
    doc->setModified();
    doc->backupItems();
    return 1;
  } else
    return 0;
}

void KplApp::newFrameItem(double x0, double w, double y0, double h,
                          double xmin, double xmax, double ymin, double ymax,
                          int logx, int logy, int ndigx, int ndigy,
                          const QString& colFrame, const QString& colGrid,
                          double xtic, double ytic, int mticx, int mticy,
                          int gridmode, int iex, int iey,
                          const QString& sx, const QString& sy,
                          const QString& sh, double relSize, int active)
{
  doc->items()->append(new FrameItem(active, logx, logy, ndigx, ndigy, mticx,
                                     mticy, gridmode, iex, iey, colFrame,
                                     colGrid, x0, w, y0, h, xmin, xmax, ymin,
                                     ymax, xtic, ytic, relSize, sx, sy, sh,
                                     &doc->aut));
  doc->setModified();
  doc->backupItems();
}

void KplApp::newAutoFrameItem()
{
  doc->items()->append(new FrameItem(&doc->aut));
  doc->setModified();
  doc->backupItems();
}

int KplApp::newArrayItem(const QString& path, int ix, int iy, int ie,
                         int istart, int n, int errbars, int symb,
                         const QString& color, double fx, double fy,
                         int active)
{
  doc->items()->append(new ArrayItem(active, symb, color, fx, fy,
                                     ix, iy, ie, istart, n, errbars, path));
  ArrayItem* it = (ArrayItem*) doc->items()->current();
  if (it->nrows) {
    doc->setModified();
    doc->backupItems();
    return 1;
  } else {
    doc->items()->remove(it);
    return 0;
  }
}

int KplApp::newFunItem(const QString& path, const QString& _name,
                       double xmin, double xmax, double dx,
                       int symb, const QString& color,
                       double fx, double fy, int active)
{
  doc->items()->append(new FunItem(active, symb, color, fx, fy,
                                   xmin, xmax, dx, _name, path));
  FunItem* it = (FunItem*) doc->items()->current();
  if (it->hmody) {
    doc->setModified();
    doc->backupItems();
    return 1;
  } else {
    doc->items()->remove(it);
    return 0;
  }
}

int KplApp::newParFunItem(const QString& pathx, const QString& namex,
                          const QString& pathy, const QString& namey,
                          double tmin, double tmax, double dt,
                          int symb, const QString& color,
                          double fx, double fy, int active)
{
  doc->items()->append(new ParFunItem(active, symb, color, fx, fy, tmin, tmax,
                                      dt, namex, pathx, namey, pathy));
  ParFunItem* it = (ParFunItem*) doc->items()->current();
  if (it->hmodx || it->hmody) {
    doc->setModified();
    doc->backupItems();
    return 1;
  } else {
    doc->items()->remove(it);
    return 0;
  }
}

void KplApp::newSplineItem(int deriv, double low, double xmin, double xmax,
                           double dx, int symb, const QString& color,
                           double relSize, double fx, double fy, int active)
{
  doc->items()->append(new SplineItem(active, symb, color, fx, fy, xmin, xmax,
                                      dx, deriv, low, relSize));
  doc->setModified();
  doc->backupItems();
}

int KplApp::newArray3DItem(double x0, double w, double y0, double h,
                           double xmin, double xmax, double ymin, double ymax,
                           double zmin, double zmax, int logx, int logy,
                           int logz, int ndigx, int ndigy, int ndigz,
                           const QString& colFrame, const QString& colGrid,
                           const QString& colData, double xtic, double ytic,
                           double ztic, int mticx, int mticy, int mticz,
                           int frame, int gridmode, int gridmode3D,
                           double xBox, double yBox, double zBox, double xRef,
                           double yRef, double phi, double theta, int iex,
                           int iey, int iez, const QString& sx,
                           const QString& sy, const QString& sz,
                           const QString& sh, double relSize,
                           const QString& path, int ix, int iy, int iz, int ie,
                           int istart, int n, int errbars, double smf,
                           double dx, double dz, double fx, double fy,
                           double fz, int active)
{
  doc->items()->append(new Array3DItem(active, logx, logy, logz, frame, ndigx,
                                       ndigy, ndigz, mticx, mticy, mticz,
                                       gridmode, gridmode3D, iex, iey, iez,
                                       colFrame, colGrid, colData, x0, w, y0,
                                       h, xmin, xmax, ymin, ymax, zmin, zmax,
                                       dx, dz, xtic, ytic, ztic, relSize, phi,
                                       theta, xBox, yBox, zBox, xRef, yRef, fx,
                                       fy, fz, sx, sy, sz, sh, ix, iy, iz, ie,
                                       istart, n, errbars, smf, path,
                                       &doc->aut));
  Array3DItem* it = (Array3DItem*) doc->items()->current();
  if (it->nrows) {
    doc->setModified();
    doc->backupItems();
    return 1;
  } else {
    doc->items()->remove(it);
    return 0;
  }
}

int KplApp::newFun3DItem(double x0, double w, double y0, double h,
                         double xmin, double xmax, double ymin, double ymax,
                         double zmin, double zmax, int logx, int logy,
                         int logz, int ndigx, int ndigy, int ndigz,
                         const QString& colFrame, const QString& colGrid,
                         const QString& colData, double xtic, double ytic,
                         double ztic, int mticx, int mticy, int mticz,
                         int frame, int gridmode, int gridmode3D, double xBox,
                         double yBox, double zBox, double xRef, double yRef,
                         double phi, double theta, int iex, int iey, int iez,
                         const QString& sx, const QString& sy,
                         const QString& sz, const QString& sh, double relSize,
                         const QString& path, const QString& _name,
                         double dx, double dz, double fx, double fy,
                         double fz, int active)
{
  doc->items()->append(new Fun3DItem(active, logx, logy, logz, frame, ndigx,
                                     ndigy, ndigz, mticx, mticy, mticz,
                                     gridmode, gridmode3D, iex, iey, iez,
                                     colFrame, colGrid, colData, x0, w, y0, h,
                                     xmin, xmax, ymin, ymax, zmin, zmax, dx,
                                     dz, xtic, ytic, ztic, relSize, phi, theta,
                                     xBox, yBox, zBox, xRef, yRef, fx, fy, fz,
                                     sx, sy, sz, sh, _name, path, &doc->aut));
  Fun3DItem* it = (Fun3DItem*) doc->items()->current();
  if (it->hmody) {
    doc->setModified();
    doc->backupItems();
    return 1;
  } else {
    doc->items()->remove(it);
    return 0;
  }
}

void KplApp::newLegendItem(int symb, const QString& colSymb, double x, double y,
                           const QString& text, const QString& colText,
                           double xoff, double yoff, double relSize,
                           int active)
{
  doc->items()->append(new LegendItem(active, symb, colSymb, colText, x, y,
                                      xoff, yoff, relSize, text));
  doc->setModified();
  doc->backupItems();
}

void KplApp::newTextItem(const QString& text, const QString& color, int align,
                         double x, double y, double dir, double relSize,
                         int active)
{
  doc->items()->append(new TextItem(active, align, color, x, y, dir, relSize,
                                    text));
  doc->setModified();
  doc->backupItems();
}

void KplApp::newLineItem(int symb, const QString& color, double x, double y,
                          double dir, double len, double relSize, int active)
{
  doc->items()->append(new LineItem(active, symb, color, x, y, dir, len,
                                    relSize));
  doc->setModified();
  doc->backupItems();
}

void KplApp::newArrowItem(const QString& color, double x, double y,
                          double dir, double len, double relSize, int active)
{
  doc->items()->append(new ArrowItem(active, color, x, y, dir, len, relSize));
  doc->setModified();
  doc->backupItems();
}

void KplApp::newArcItem(const QString& color, double x, double y, double w,
                        double h, double a, double alen, double relSize,
                        double ang, int active)
{
  doc->items()->append(new ArcItem(active, color, x, y, w, h, a, alen,
                                   relSize, ang));
  doc->setModified();
  doc->backupItems();
}
                  
void KplApp::newRectangleItem(const QString& color, double x, double y,
                              double w, double h, double relSize, int fill,
                              int active)
{
  doc->items()->append(new RectItem(active, fill, color, x, y, w, h, relSize));
  doc->setModified();
  doc->backupItems();
}

void KplApp::newEllipseItem(const QString& color, double x, double y,
                            double w, double h, double relSize, int fill,
                            int active)
{
  doc->items()->append(new EllipseItem(active, fill, color, x, y, w, h,
                                       relSize));
  doc->setModified();
  doc->backupItems();
}

int KplApp::autoScaleFrame(int iItem, int autoNorm)
{
  if (unsigned(iItem) >= doc->items()->count())
    return 0;
  if (doc->items()->at(iItem)->iType() != KplItem::Frame)
    return 0;
  double fx, fy;
  int iex, iey;
  FrameItem* it = (FrameItem*) doc->items()->at(iItem);
  if (autoNorm)
    it->logx = it->logy = false;
  doc->autoScaleFrame(autoNorm, iItem, &iex, &iey, &fx, &fy, &it->xmi, &it->xma,
                      &it->ymi, &it->yma, &it->xtic, &it->ytic,
                      &it->mticx, &it->mticy, &it->ndigx, &it->ndigy,
                      it->w / it->h, it->logx, it->logy);
  if ((iex != 999) && (iey != 999)) {
    it->iex = iex;
    it->iey = iey;
    for (KplItem* itm = doc->items()->at(iItem + 1); itm;
         itm = doc->items()->next()) {
      if (itm->iType() == KplItem::Frame)
        break;
      itm->normalize(fx, fy);
    }
    doc->setModified();
    doc->backupItems();
    return 1;
  } else
    return 0;
}

int KplApp::loadPar(const QString& url, int iItem, int yFun)
{
  int success = false;
  if (unsigned(iItem) >= doc->items()->count())
    return success;
  KplItem::ItemTypes ityp = doc->items()->at(iItem)->iType();
  if ((ityp != KplItem::Function) && (ityp != KplItem::ParFunction) &&
      (ityp != KplItem::Function3D))
    return success;
  double *p;
  if ((ityp == KplItem::Function) || (ityp == KplItem::Function3D))
    if (!yFun)
      return success;
    else
      if (ityp == KplItem::Function)
        p = ((FunItem*) doc->items()->at(iItem))->py;
      else
        p = ((Fun3DItem*) doc->items()->at(iItem))->py;
  else
    p = yFun ? ((ParFunItem*) doc->items()->at(iItem))->py :
               ((ParFunItem*) doc->items()->at(iItem))->px;
  success = Utils::loadPar(KURL(url), this, p, doc) ? 1 : 0;
  if (success) {
    doc->setModified();
    doc->backupItems();
  }
  return success;
}

int KplApp::setPar(int iItem, int iPar, double value, int yFun)
{
  if ((unsigned(iItem) >= doc->items()->count()) ||
      (iPar < 0) || (iPar >= KPL_NPMAX))
    return 0;
  KplItem::ItemTypes ityp = doc->items()->at(iItem)->iType();
  if ((ityp != KplItem::Function) && (ityp != KplItem::ParFunction) &&
      (ityp != KplItem::Function3D))
    return 0;
  if (((ityp == KplItem::Function) || (ityp == KplItem::Function3D)) &&
      (!yFun))
    return 0;
  doc->items()->at(iItem)->setPar(iPar, value, yFun);
  doc->setModified();
  doc->backupItems();
  return 1;
}

int KplApp::setArrayRange(int iItem, int iStart, int n)
{
  if (unsigned(iItem) >= doc->items()->count())
    return 0;
  KplItem::ItemTypes ityp = doc->items()->at(iItem)->iType();
  if ((ityp != KplItem::Array) && (ityp != KplItem::Array3D))
    return 0;
  if (ityp == KplItem::Array) {
    ArrayItem* it = (ArrayItem*) doc->items()->at(iItem);
    if ((iStart < 0) || (iStart > (it->nrows - 1)) || (n < 1) ||
        ((iStart + n) > it->nrows))
      return 0;
    it->istart = iStart;
    it->n = n;
  } else {
    Array3DItem* it = (Array3DItem*) doc->items()->at(iItem);
    if ((iStart < 0) || (iStart > (it->nrows - 1)) || (n < 1) ||
        ((iStart + n) > it->nrows))
      return 0;
    it->istart = iStart;
    it->n = n;
  }
  doc->setModified();
  doc->backupItems();
  return 1;
}

int KplApp::addFitItems(int erase, int iArr, int iFun, int errCol,
                        const QString& errModPath, const QString& errModName)
{
  if ((unsigned(iArr) >= doc->items()->count()) ||
      (unsigned(iFun) >= doc->items()->count()))
    return 0;
  if ((doc->items()->at(iArr)->iType() != KplItem::Array) ||
      (doc->items()->at(iFun)->iType() != KplItem::Function))
    return 0;
  if (erase) {
    arr.clear();
    fun.clear();
  }
  QString s;
  s.sprintf("DataErrors%i", arr.count());
  config->setGroup(s);
  config->writeEntry("FitErrorColumn", bool(errCol));
  if (!errCol) {
    config->writeEntry("ErrorModelPath", errModPath);
    config->writeEntry("ErrorModelName", errModName);
  }
  arr.append((ArrayItem*) doc->items()->at(iArr));
  fun.append((FunItem*) doc->items()->at(iFun));
  return 1;
}

int KplApp::fit(int nonLin, int savePar, int follow)
{
  doc->chisq = 0.0;
  if (!arr.count())
    return 0;
  doc->aut.fitNonLin = nonLin;
  int mode = 0;
  if (savePar)
    mode |= FitDlg::SavePar;
  if (follow)
    mode |= FitDlg::Follow;
  FitDlg dlg(this, doc, &arr, &fun, mode);
  dlg.fit();
  dlg.getValues(false);
  return 1;
}

int KplApp::splineFit(int iArr, int iSpl, int deg, double smf, double xmin,
                      double xmax, int errCol, const QString& errModPath,
                      const QString& errModName)
{
  if ((doc->items()->at(iArr)->iType() != KplItem::Array) ||
      (doc->items()->at(iSpl)->iType() != KplItem::Spline))
    return 0;
  doc->chisq = 0.0;
  SplFitDlg dlg(this, doc, (ArrayItem*) doc->items()->at(iArr),
                (SplineItem*)  doc->items()->at(iSpl), 0);
  if (dlg.fit(deg, smf, xmin, xmax, errCol, errModPath, errModName) < 4) {
    dlg.getValues(false);
    return 1;
  } else
    return 0;
}

double KplApp::chiSquare()
{
  return doc->chisq;
}

int KplApp::saveFitPar(const QString& url)
{
  return Utils::saveParameter(this, KURL(url), doc->pLast,
                              doc->pErr, doc) ? 1 : 0;
}

int KplApp::exportValues(const QString& url, int iItem)
{
  KplItem::ItemTypes ityp = doc->items()->at(iItem)->iType();
  if ((ityp != KplItem::Function) && (ityp != KplItem::ParFunction) &&
      (ityp != KplItem::Function3D) && (ityp != KplItem::Array3D))
    return 0;
  KURL u = KURL(url);
  QFile f(u.isLocalFile() ? u.path() : doc->tmpFile());
  int success = f.open(IO_WriteOnly) ? 1 : 0;
  if (success) {
    QTextStream ts(&f);
    doc->items()->at(iItem)->exportTable(ts, doc);
    f.close();
    if (!u.isLocalFile())
      doc->copyTmp(u);
  } else
    KMessageBox::sorry(this, i18n("Writing to this file not possible."));
  return success;
}
