// Copyright The OpenTelemetry Authors
// SPDX-License-Identifier: Apache-2.0

package ebpfcommon

import (
	"bufio"
	"bytes"
	"errors"
	"io"
	"log/slog"
	"net"
	"net/http"
	"strconv"
	"strings"

	"go.opentelemetry.io/obi/pkg/appolly/app/request"
	ebpfhttp "go.opentelemetry.io/obi/pkg/ebpf/common/http"
	"go.opentelemetry.io/obi/pkg/internal/ebpf/ringbuf"
)

func removeQuery(url string) string {
	idx := strings.IndexByte(url, '?')
	if idx > 0 {
		return url[:idx]
	}
	return url
}

type HTTPInfo struct {
	BPFHTTPInfo
	Method     string
	URL        string
	Host       string
	Peer       string
	HeaderHost string
	Body       string
}

// misses serviceID
func httpInfoToSpanLegacy(info *HTTPInfo) request.Span {
	scheme := "http"
	if info.Ssl == 1 {
		scheme = "https"
	}

	return request.Span{
		Type:           request.EventType(info.Type),
		Method:         info.Method,
		Path:           removeQuery(info.URL),
		Peer:           info.Peer,
		PeerPort:       int(info.ConnInfo.S_port),
		Host:           info.Host,
		HostPort:       int(info.ConnInfo.D_port),
		ContentLength:  int64(info.Len),
		ResponseLength: int64(info.RespLen),
		RequestStart:   int64(info.ReqMonotimeNs),
		Start:          int64(info.StartMonotimeNs),
		End:            int64(info.EndMonotimeNs),
		Status:         int(info.Status),
		TraceID:        info.Tp.TraceId,
		SpanID:         info.Tp.SpanId,
		ParentSpanID:   info.Tp.ParentId,
		TraceFlags:     info.Tp.Flags,
		Pid: request.PidInfo{
			HostPID:   info.Pid.HostPid,
			UserPID:   info.Pid.UserPid,
			Namespace: info.Pid.Ns,
		},
		Statement: scheme + request.SchemeHostSeparator + info.HeaderHost,
	}
}

func httpRequestResponseToSpan(parseCtx *EBPFParseContext, event *BPFHTTPInfo, req *http.Request, resp *http.Response) request.Span {
	defer req.Body.Close()
	defer resp.Body.Close()

	peer, host := (*BPFConnInfo)(&event.ConnInfo).reqHostInfo()

	httpSpan := request.Span{
		Type:           request.EventType(event.Type),
		Method:         req.Method,
		Path:           removeQuery(req.URL.String()),
		Peer:           peer,
		PeerPort:       int(event.ConnInfo.S_port),
		Host:           host,
		HostPort:       int(event.ConnInfo.D_port),
		ContentLength:  req.ContentLength,
		ResponseLength: resp.ContentLength,
		RequestStart:   int64(event.ReqMonotimeNs),
		Start:          int64(event.StartMonotimeNs),
		End:            int64(event.EndMonotimeNs),
		Status:         resp.StatusCode,
		TraceID:        event.Tp.TraceId,
		SpanID:         event.Tp.SpanId,
		ParentSpanID:   event.Tp.ParentId,
		TraceFlags:     event.Tp.Flags,
		Pid: request.PidInfo{
			HostPID:   event.Pid.HostPid,
			UserPID:   event.Pid.UserPid,
			Namespace: event.Pid.Ns,
		},
		Statement: req.URL.Scheme + request.SchemeHostSeparator + req.Host,
	}

	if isClientEvent(event.Type) && parseCtx != nil && parseCtx.payloadExtraction.HTTP.AWS.Enabled {
		span, ok := ebpfhttp.AWSS3Span(&httpSpan, req, resp)
		if ok {
			return span
		}

		span, ok = ebpfhttp.AWSSQSSpan(&httpSpan, req, resp)
		if ok {
			return span
		}
	}

	if !isClientEvent(event.Type) && parseCtx != nil && parseCtx.payloadExtraction.HTTP.GraphQL.Enabled {
		span, ok := ebpfhttp.GraphQLSpan(&httpSpan, req, resp)
		if ok {
			return span
		}
	}

	if isClientEvent(event.Type) && parseCtx != nil && parseCtx.payloadExtraction.HTTP.Elasticsearch.Enabled {
		span, ok := ebpfhttp.ElasticsearchSpan(&httpSpan, req, resp)
		if ok {
			return span
		}
	}

	return httpSpan
}

func ReadHTTPInfoIntoSpan(parseCtx *EBPFParseContext, record *ringbuf.Record, filter ServiceFilter) (request.Span, bool, error) {
	event, err := ReinterpretCast[BPFHTTPInfo](record.RawSample)
	if err != nil {
		return request.Span{}, true, err
	}

	// Generated by Go instrumentation
	if !filter.ValidPID(event.Pid.UserPid, event.Pid.Ns, PIDTypeKProbes) {
		return request.Span{}, true, nil
	}

	return HTTPInfoEventToSpan(parseCtx, event)
}

func HTTPInfoEventToSpan(parseCtx *EBPFParseContext, event *BPFHTTPInfo) (request.Span, bool, error) {
	var (
		requestBuffer, responseBuffer []byte
		hasResponse                   bool
		isClient                      = isClientEvent(event.Type)
	)

	if event.HasLargeBuffers == 1 {
		b, ok := extractTCPLargeBuffer(parseCtx, event.Tp.TraceId, packetTypeRequest, directionByPacketType(packetTypeRequest, isClient), event.ConnInfo)
		if ok {
			requestBuffer = b
		} else {
			slog.Debug("missing large buffer for HTTP request", "traceID", event.Tp.TraceId, "conn", event.ConnInfo, "packetType", packetTypeRequest)
		}

		b, ok = extractTCPLargeBuffer(parseCtx, event.Tp.TraceId, packetTypeResponse, directionByPacketType(packetTypeResponse, isClient), event.ConnInfo)
		if ok {
			responseBuffer = b
			hasResponse = true
		} else {
			slog.Debug("missing large buffer for HTTP response", "traceID", event.Tp.TraceId, "conn", event.ConnInfo, "packetType", packetTypeResponse)
		}
	} else {
		requestBuffer = event.Buf[:]
	}

	if parseCtx != nil && !parseCtx.payloadExtraction.Enabled() {
		// There's no need to parse HTTP headers/body,
		// create the span directly.
		return httpRequestToSpan(event, requestBuffer), false, nil
	}

	if !hasResponse {
		// Large buffers disabled
		return httpRequestToSpan(event, requestBuffer), false, nil
	}

	req, err := http.ReadRequest(bufio.NewReader(bytes.NewReader(requestBuffer)))
	resp, err2 := httpSafeParseResponse(responseBuffer, req)
	if err != nil || err2 != nil {
		slog.Debug("error while parsing http request or response, falling back to manual HTTP info parsing", "reqErr", err, "respErr", err2)
		return httpRequestToSpan(event, requestBuffer), false, nil
	}

	return httpRequestResponseToSpan(parseCtx, event, req, resp), false, nil
}

// HTTP response buffers might have been sent incomplete, before the full body.
// Try to parse the original buffer first, if an EOF is encountered, append an empty
// body to the buffer and try again.
func httpSafeParseResponse(responseBuffer []byte, req *http.Request) (*http.Response, error) {
	rd := bufio.NewReader(bytes.NewReader(responseBuffer))
	resp, err := http.ReadResponse(rd, req)
	if err != nil && errors.Is(err, io.ErrUnexpectedEOF) {
		// Append empty body and try again
		responseBuffer := append(responseBuffer, []byte("\r\n\r\n")...)
		rd = bufio.NewReader(bytes.NewReader(responseBuffer))
		return http.ReadResponse(rd, req)
	}
	return resp, nil
}

func httpRequestToSpan(event *BPFHTTPInfo, requestBuffer []byte) request.Span {
	var (
		result     = HTTPInfo{BPFHTTPInfo: *event}
		bufHost    string
		bufPort    int
		parsedHost bool
	)

	// When we can't find the connection info, we signal that through making the
	// source and destination ports equal to max short. E.g. async SSL
	if event.ConnInfo.S_port != 0 || event.ConnInfo.D_port != 0 {
		source, target := (*BPFConnInfo)(&event.ConnInfo).reqHostInfo()
		result.Host = target
		result.Peer = source
	} else {
		bufHost, bufPort = httpHostFromBuf(requestBuffer)
		parsedHost = true

		if bufPort >= 0 {
			result.Host = bufHost
			result.ConnInfo.D_port = uint16(bufPort)
		}
	}
	result.URL = httpURLFromBuf(requestBuffer)
	result.Method = httpMethodFromBuf(requestBuffer)

	if request.EventType(result.Type) == request.EventTypeHTTPClient && !parsedHost {
		bufHost, _ = httpHostFromBuf(requestBuffer)
	}

	result.HeaderHost = bufHost

	return httpInfoToSpanLegacy(&result)
}

func httpURLFromBuf(req []byte) string {
	buf := string(req)
	space := strings.Index(buf, " ")
	if space < 0 {
		return ""
	}

	bufEnd := bytes.IndexByte(req, 0) // We assume the buffer was zero initialized in eBPF
	if bufEnd < 0 {
		bufEnd = len(buf)
	}

	if space+1 > bufEnd {
		return ""
	}

	nextSpace := strings.IndexAny(buf[space+1:bufEnd], " \r\n")
	if nextSpace < 0 {
		return buf[space+1 : bufEnd]
	}

	end := min(nextSpace+space+1, bufEnd)

	return buf[space+1 : end]
}

func httpMethodFromBuf(req []byte) string {
	buf := string(req)
	space := strings.Index(buf, " ")
	if space < 0 {
		return ""
	}

	return buf[:space]
}

func httpHostFromBuf(req []byte) (string, int) {
	buf := cstr(req)

	host := "Host: "
	idx := strings.Index(buf, host)

	if idx < 0 {
		return "", -1
	}

	buf = buf[idx+len(host):]

	rIdx := strings.Index(buf, "\r")

	// only parse full host information, partial may
	// get the wrong name or wrong port
	if rIdx < 0 {
		return "", -1
	}

	host, portStr, err := net.SplitHostPort(buf[:rIdx])
	if err != nil {
		return buf[:rIdx], -1
	}

	port, _ := strconv.Atoi(portStr)

	return host, port
}
