// This may look like C code, but it is really -*- C++ -*-
/* 
Copyright (C) 1988 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)
    based on code by Marc Shapiro (shapiro@sor.inria.fr)

This file is part of GNU CC.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the GNU CC General Public
License for full details.

Everyone is granted permission to copy, modify and redistribute
GNU CC, but only under the conditions described in the
GNU CC General Public License.   A copy of this license is
supposed to have been given to you along with GNU CC so you
can know your rights and responsibilities.  It should be in a
file named COPYING.  Among other things, the copyright notice
and this notice must be preserved on all copies.  
*/

#include "<T>.XPlex.h"


<T>XPlex:: <T>XPlex()
{
  mods = 0;
  lo = fnc = 0;
  csize = DEFAULT_INITIAL_CAPACITY;
  <T>* data = new <T>[csize];
  hd = ch = new <T>IChunk(data,  lo, lo, fnc, lo+csize);
}

<T>XPlex:: <T>XPlex(int chunksize)
{
  if (chunksize == 0) error("invalid constructor specification");
  mods = 0;
  lo = fnc = 0;
  if (chunksize > 0)
  {
    csize = chunksize;
    <T>* data = new <T>[csize];
    hd = ch = new <T>IChunk(data,  lo, lo, fnc, csize);
  }
  else
  {
    csize = -chunksize;
    <T>* data = new <T>[csize];
    hd = ch = new <T>IChunk(data,  chunksize, lo, fnc, fnc);
  }
}


<T>XPlex:: <T>XPlex(int l, int chunksize)
{
  if (chunksize == 0) error("invalid constructor specification");
  mods = 0;
  lo = fnc = l;
  if (chunksize > 0)
  {
    csize = chunksize;
    <T>* data = new <T>[csize];
    hd = ch = new <T>IChunk(data,  lo, lo, fnc, csize+lo);
  }
  else
  {
    csize = -chunksize;
    <T>* data = new <T>[csize];
    hd = ch = new <T>IChunk(data,  chunksize+lo, lo, fnc, fnc);
  }
}

void <T>XPlex::make_initial_chunks(int up = 1)
{
  int need = fnc - lo;
  hd = 0;
  if (up)
  {
    int l = lo;
    do
    {
      int sz;
      if (need >= csize)
        sz = csize;
      else
        sz = need;
      <T>* data = new <T> [csize];
      <T>IChunk* h = new <T>IChunk(data,  l, l, l+sz, l+csize);
      if (hd != 0)
        h->link_to_next(hd);
      else
        hd = h;
      l += sz;
      need -= sz;
    } while (need > 0);
  }
  else
  {
    int hi = fnc;
    do
    {
      int sz;
      if (need >= csize)
        sz = csize;
      else
        sz = need;
      <T>* data = new <T> [csize];
      <T>IChunk* h = new <T>IChunk(data,  hi-csize, hi-sz, hi, hi);
      if (hd != 0)
        h->link_to_next(hd);
      hd = h;
      hi -= sz;
      need -= sz;
    } while (need > 0);
  }
  ch = hd;
}

<T>XPlex:: <T>XPlex(int l, int hi, const <T&> initval, int chunksize = 0)
{
  mods = 0;
  lo = l;
  fnc = hi + 1;
  if (chunksize == 0)
  {
    csize = fnc - l;
    make_initial_chunks(1);
  }
  else if (chunksize < 0)
  {
    csize = -chunksize;
    make_initial_chunks(0);
  }
  else
  {
    csize = chunksize;
    make_initial_chunks(1);
  }
  fill(initval);
}

<T>XPlex::<T>XPlex(const <T>XPlex& a)
{
  mods = 0;
  lo = a.lo;
  fnc = a.fnc;
  csize = a.csize;
  make_initial_chunks();
  for (int i = a.low(); i < a.fence(); a.next(i)) (*this)[i] = a[i];
}

void <T>XPlex::operator= (const <T>XPlex& a)
{
  if (&a != this)
  {
    invalidate();
    record_change();
    lo = a.lo;
    fnc = a.fnc;
    csize = a.csize;
    make_initial_chunks();
    for (int i = a.low(); i < a.fence(); a.next(i)) (*this)[i] = a[i];
  }
}


void <T>XPlex::cache(int idx)
{
  <T>IChunk* tail = tl();
  while (idx >= ch->fence_index())
  {
    if (ch == tail) index_error();
    ch = ch->next();
  }
  while (idx < ch->low_index())
  {
    if (ch == hd) index_error();
    ch = ch->prev();
  }
}


void <T>XPlex::cache(const <T>* p)
{
  <T>IChunk* old = ch;
  while (!ch->actual_pointer(p))
  {
    ch = ch->next();
    if (ch == old) index_error();
  }
}

int <T>XPlex::owns(Pix p)
{
  <T>IChunk* old = ch;
  while (!ch->actual_pointer(p))
  {
    ch = ch->next();
    if (ch == old) return 0;
  }
  return 1;
}


<T>* <T>XPlex::dosucc(<T>* p)
{
  if (p == 0) return 0;
  <T>IChunk* old = ch;
  while (!ch->actual_pointer(p))
  {
    ch = ch->next();
    if (ch == old) return 0;
  }
  int i = ch->index_of(p) + 1;
  if (i >= fnc) return 0;
  if (i >= ch->fence_index()) ch = ch->next();
  return (ch->pointer_to(i));
}

<T>* <T>XPlex::dopred(<T>* p)
{
  if (p == 0) return 0;
  <T>IChunk* old = ch;
  while (!ch->actual_pointer(p))
  {
    ch = ch->prev();
    if (ch == old) return 0;
  }
  int i = ch->index_of(p) - 1;
  if (i < lo) return 0;
  if (i < ch->low_index()) ch = ch->prev();
  return (ch->pointer_to(i));
}


int <T>XPlex::add_high(const <T&> elem)
{
  record_change();
  ch =  tl();
  if (!ch->can_grow_high())
  {
    <T>* data = new <T> [csize];
    ch = new <T>IChunk(data,  fnc, fnc, fnc,fnc+csize);
    ch->link_to_prev(tl());
  }
  *((ch-><T>IChunk::grow_high())) = elem;
  return fnc++;
}

int <T>XPlex::del_high ()
{
  if (empty()) empty_error();
  record_change();
  ch =  tl();
  ch-><T>IChunk::shrink_high();
  if (ch-><T>IChunk::empty() && !one_chunk())
  {
    <T>IChunk* pred = ch->prev();
    del_chunk(ch);
    ch = pred;
  }
  return --fnc - 1;
}

int <T>XPlex::add_low (const <T&> elem)
{
  record_change();
  ch =  hd;
  if (!ch->can_grow_low())
  {
    <T>* data = new <T> [csize];
    hd = new <T>IChunk(data,  lo-csize, lo, lo, lo);
    hd->link_to_next(ch);
    ch =  hd;
  }
  *((ch-><T>IChunk::grow_low())) = elem;
  return --lo;
}


int <T>XPlex::del_low ()
{
  if (empty()) empty_error();
  record_change();
  ch =  hd;
  ch-><T>IChunk::shrink_low();
  if (ch-><T>IChunk::empty() && !one_chunk())
  {
    hd = ch->next();
    del_chunk(ch);
    ch =  hd;
  }
  return ++lo;
}

void <T>XPlex::append (const <T>Plex& a)
{
  for (int i = a.low(); i < a.fence(); a.next(i)) add_high(a[i]);
}

void <T>XPlex::prepend (const <T>Plex& a)
{
  for (int i = a.high(); i > a.ecnef(); a.prev(i)) add_low(a[i]);
}

void <T>XPlex::reverse()
{
  <T> tmp;
  int l = lo;
  int h = fnc - 1;
  ch = hd;
  <T>IChunk* hich = tl();
  while (l < h)
  {
    <T>* lptr = ch->pointer_to(l);
    <T>* hptr = hich->pointer_to(h);
    tmp = *lptr;
    *lptr = *hptr;
    *hptr = tmp;
    if (++l >= ch->fence_index()) ch = ch->next();
    if (--h < hich->low_index()) hich = hich->prev();
  }
}

void <T>XPlex::fill(<T&> x)
{
  for (int i = lo; i < fnc; ++i) (*this)[i] = x;
}

void <T>XPlex::fill(<T&> x, int l, int hi)
{
  for (int i = l; i <= hi; ++i) (*this)[i] = x;
}


void <T>XPlex::clear()
{
  if (fnc != lo)
  {
    record_change();
    ch = tl();
    while (ch != hd)
    {
      <T>IChunk* prv = ch->prev();
      del_chunk(ch);
      ch = prv;
    }
    ch-><T>IChunk::clear(lo);
    fnc = lo;
  }
}


int <T>XPlex::OK ()
{
  int v = hd != 0 && ch != 0;     // at least one chunk

  v &= fnc == tl()->fence_index();// last chunk fence == plex fence
  v &= lo == ((hd))-><T>IChunk::low_index();    // first lo == plex lo

// loop for others:
  int found_ch = 0;                   // to make sure ch is in list;
  <T>IChunk* t = (hd);
  for (;;)
  {
    if (t == ch) ++found_ch;
    v &= t-><T>IChunk::OK();              // each chunk is OK
    if (t == tl())
      break;
    else                              // and has indices contiguous to succ
    {
      v &= t->top_index() == t->next()->base_index();
      if (t != hd)                  // internal chunks full
      {
        v &= !t->empty();
        v &= !t->can_grow_low();
        v &= !t->can_grow_high();
      }
      t = t->next();
    }
  }
  v &= found_ch == 1;
  if (!v) error("invariant failure");
  return v;
}
