/* 
Copyright (C) 1988 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)

This file is part of GNU CC.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the GNU CC General Public
License for full details.

Everyone is granted permission to copy, modify and redistribute
GNU CC, but only under the conditions described in the
GNU CC General Public License.   A copy of this license is
supposed to have been given to you along with GNU CC so you
can know your rights and responsibilities.  It should be in a
file named COPYING.  Among other things, the copyright notice
and this notice must be preserved on all copies.  
*/

/*
 Conversions from class types from and to char*'s.
 This file is also used in order to place all inlined functions in libg++.a
*/

#include <stdarg.h> 
#include <ctype.h>
#include "libconfig.h"
#include <builtin.h>
#include <File.h>
#include <stream.h>
#include <PlotFile.h>
#include <SFile.h>
#include <Obstack.h>
#include <String.h>
#include <Integer.h>
#include <Rational.h>
#include <Complex.h>
#include <BitSet.h>
#include <BitString.h>
// Obstacks are used as an easy way to allocate enough space
// for various input and output & conversion operations
// (they are a real natural for input; we might as well use
// them for output too rather than creating yet another allocation
// mechanism.)
// 
// We guarantee that ONLY the most recently constructed
// obstack object is intact. There is no queuing mechanism.
// This is a firmer policy than using a queue that may or may
// not have enough space to hold several objects.

static Obstack ob;
static char* oblast = 0;

static void kill_last()
{
  if (oblast != 0)
    ob.free(oblast);
}
    

char* form(const char* fmt ...)
{
  va_list args;
  va_start(args, fmt);
  kill_last();
  ob.blank(BUFSIZ);
  oblast = (char*)(ob.finish());
  char* obbase = oblast;
#ifndef HAVE_VPRINTF
  FILE b;
  b._flag = _IOWRT|_IOSTRG;
  b._ptr = obbase;
  b._cnt = BUFSIZ;
  _doprnt(fmt, args, &b);
  putc('\0', &b);
#else
  vsprintf(obbase, fmt, args);
#endif
  va_end(args);
  return obbase;
}

char* itoa(long x, int base = 10, int width = 0)
{
  kill_last();
  int wrksiz = 100 + width;
  ob.blank(wrksiz);
  oblast = (char*)(ob.finish());
  char* obbase = oblast;
  char* e = obbase + wrksiz - 1;
  char* s = e;
  *--s = 0;
  char sgn = 0;

  if (x == 0)
    *--s = '0';
  else
  {
    int z;
    if (x < 0)
    {
      sgn = '-';
      z = -x;
    }
    else
      z = x;
    while (z != 0)
    {
      char ch = z % base;
      z = z / base;
      if (ch >= 10)
        ch += 'a' - 10;
      else
        ch += '0';
      *--s = ch;
    }
  }

  if (sgn) *--s = sgn;
  int w = e - s - 1;
  while (w++ < width)
    *--s = ' ';
  return s;
}

char* hex(long i, int width = 0)
{
  return itoa(i, 16, width);
}

char* oct(long i, int width = 0)
{
  return itoa(i, 8, width);
}

char* dec(long i, int width = 0)
{
  return itoa(i, 10, width);
}


Integer atoI(const char* s)
{
  Integer r = 0;
  if (s != 0)
  {
    while (s != 0 && isspace(*s));
    if (s != 0)
    {
      char sgn = I_POSITIVE;
      if (*s == '-')
      {
        sgn = I_NEGATIVE;
        s++;
      }
      while (s != 0 && *s >= '0' && *s <= '9')
      {
        long digit = *s - '0';
        multiply(r, 10, r);
        r += digit;
        ++s;
      }
      r.rep->sgn = sgn;
    }
  }
  return r;
}


char* Itoa(Integer& x, int base = 10, int width = 0)
{
  kill_last();

  int wrksiz = (x.rep->len + 1) * I_SHIFT * 4 + 1 + width;
  ob.blank(wrksiz);
  oblast = (char*)(ob.finish());
  char* obbase = oblast;
  char* e = obbase + wrksiz - 1;
  char* s = e;
  *--s = 0;

  int xsgn = sign(x);
  if (xsgn == 0)
    *--s = '0';
  else
  {
    Integer z = abs(x);

    // split division by base into two parts: 
    // first divide by biggest power of base that fits in an unsigned short,
    // then use straight signed div/mods from there. 

    // find power
    int bpower = 1;
    unsigned short b = base;
    unsigned short maxb = I_MAXNUM / base;
    while (b < maxb)
    {
      b *= base;
      ++bpower;
    }

    int rem;

    for(;;)
    {
      divide(z, b, z, rem);
      if (z.rep->len == 0)
      {
        while (rem != 0)
        {
          char ch = rem % base;
          rem /= base;
          if (ch >= 10)
            ch += 'a' - 10;
          else
            ch += '0';
          *--s = ch;
        }
        break;
      }
      else
      {
        for (int i = 0; i < bpower; ++i)
        {
          char ch = rem % base;
          rem /= base;
          if (ch >= 10)
            ch += 'a' - 10;
          else
            ch += '0';
          *--s = ch;
        }
      }
    }
  }

  if (xsgn < 0) *--s = '-';
  int w = e - s - 1;
  while (w++ < width)
    *--s = ' ';
  return s;
}

char* dec(Integer& x, int width = 0)
{
  return Itoa(x, 10, width);
}

char* oct(Integer& x, int width = 0)
{
  return Itoa(x, 8, width);
}

char* hex(Integer& x, int width = 0)
{
  return Itoa(x, 16, width);
}

istream& operator >> (istream& s, Integer& y)
{
  int got_one = 0;
  if (!s.readable())
  {
    s.error();
    return s;
  }

  kill_last();

  char sgn = 0;
  char ch;
  s >> WS;
  while (s.good())
  {
    s.get(ch);
    if (ch == '-')
    {
      if (sgn == 0)
      {
        sgn = 1;
        ob.grow(ch);
      }
      else
        break;
    }
    else if (ch >= '0' && ch <= '9')
    {
      got_one = 1;
      ob.grow(ch);
    }
    else
      break;
  }
  char * p = (char*)(ob.finish(0));
  oblast = p;
  if (s.good())
    s.unget(ch);
  if (!got_one)
    s.error();
  else
    y = atoI(p);
  return s;
}


const char* BitSettoa(BitSet& x, char f = '0', char t = '1', char star = '*')
{
  kill_last();

  x.trim();

  unsigned long* s = x.rep->s;
  unsigned long* top = &(s[x.rep->len - 1]);

  while (s < top)
  {
    unsigned long a = *s++;
    for (int j = 0; j < B_SHIFT; ++j)
    {
      ob.grow((a & 1)? t : f);
      a >>= 1;
    }
  }

  if (!x.rep->virt)
  {
    unsigned long a = *s;
    for (int j = 0; j < B_SHIFT && a != 0; ++j)
    {
      ob.grow((a & 1)? t : f);
      a >>= 1;
    }
    ob.grow(f);
  }
  else
  {
    unsigned long a = *s;
    unsigned long mask = ~(0L);
    unsigned long himask = (1 << (B_SHIFT - 1)) - 1;
    for (int j = 0; j < B_SHIFT && a != mask; ++j)
    {
      ob.grow((a & 1)? t : f);
      a = (a >> 1) & himask;
      mask = (mask >> 1) & himask;
    }
    ob.grow(t);
  }

  ob.grow(star);

  return oblast = (char*)(ob.finish(0));
}

BitSet atoBitSet(const char* s, char f = '0', char t = '1', char star = '*')
{
  BitSet r;
  int sl = strlen(s);
  if (sl != 0)
  {
    r.setlength(sl / B_SHIFT + 1, 1);
    unsigned long* rs = r.rep->s;
    unsigned long a = 0;
    unsigned long m = 1;
    char lastch = 0;
    int i = 0;
    int l = 1;
    for(;;)
    {
      char ch = s[i];
      if (ch == t)
        a |= m;
      else if (ch == star)
      {
        if (r.rep->virt = lastch == t)
          *rs = a | ~(m - 1);
        else
          *rs = a;
        break;
      }
      else if (ch != f)
      {
        *rs = a;
        break;
      }
      lastch = ch;
      if (++i == sl)
      {
        *rs = a;
        break;
      }
      else if (i % B_SHIFT == 0)
      {
        *rs++ = a;
        a = 0;
        m = 1;
        ++l;
      }
      else
        m <<= 1;
    }
    r.rep->len = l;
    r.trim();
  }
  return r;
}

ostream& operator << (ostream& s, BitSet& x)
{
  return s << BitSettoa(x);
}


const char* BitStringtoa(BitString& x, char f = '0', char t = '1')
{
  kill_last();
  int xl = x.rep->len;
  unsigned long* s = x.rep->s;
  unsigned long a;

  for (int i = 0; i < xl; ++i)
  {
    if (i % B_SHIFT == 0)
      a = *s++;
    ob.grow((a & 1)? t : f);
    a >>= 1;
  }

  return oblast = (char*)(ob.finish(0));
}

BitString atoBitString(const char* s, char f = '0', char t = '1')
{
  BitString r;
  int sl = strlen(s);
  if (sl != 0)
  {
    int rl = 0;
    r.setlength(sl);
    unsigned long* rs = r.rep->s;
    unsigned long a = 0;
    unsigned long m = 1;
    int i = 0;
    for(;;)
    {
      char ch = s[i];
      if (ch != t && ch != f)
      {
        *rs = a;
        break;
      }
      ++rl;
      if (ch == t)
        a |= m;
      if (++i == sl)
      {
        *rs = a;
        break;
      }
      else if (i % B_SHIFT == 0)
      {
        *rs++ = a;
        a = 0;
        m = 1;
      }
      else
        m <<= 1;
    }
    r.setlength(rl);
  }
  return r;
}

ostream& operator << (ostream& s, BitString& x)
{
  return s << BitStringtoa(x);
}


const char* BitPatterntoa(BitPattern& p, char f = '0',char t = '1',char x='X')
{
  kill_last();
  int pl = p.pattern.rep->len;
  int ml = p.mask.rep->len;
  int l = pl <? ml;
  unsigned long* ps = p.pattern.rep->s;
  unsigned long* ms = p.mask.rep->s;
  unsigned long a;
  unsigned long m;

  for (int i = 0; i < l; ++i)
  {
    if (i % B_SHIFT == 0)
    {
      a = *ps++;
      m = *ms++;
    }
    if (m & 1)
      ob.grow((a & 1)? t : f);
    else
      ob.grow(x);
    a >>= 1;
    m >>= 1;
  }

  return oblast = (char*)(ob.finish(0));
}

BitPattern atoBitPattern(const char* s,char f = '0',char t = '1',char x = 'X')
{
  BitPattern r;
  int sl = strlen(s);
  if (sl != 0)
  {
    int rl = 0;
    r.pattern.setlength(sl);
    r.mask.setlength(sl);
    unsigned long* rs = r.pattern.rep->s;
    unsigned long* ms = r.mask.rep->s;
    unsigned long a = 0;
    unsigned long b = 0;
    unsigned long m = 1;
    int i = 0;
    for(;;)
    {
      char ch = s[i];
      if (ch != t && ch != f && ch != x)
      {
        *rs = a;
        *ms = b;
        break;
      }
      ++rl;
      if (ch == t)
      {
        a |= m;
        b |= m;
      }
      else if (ch == f)
      {
        b |= m;
      }
      if (++i == sl)
      {
        *rs = a;
        *ms = b;
        break;
      }
      else if (i % B_SHIFT == 0)
      {
        *rs++ = a;
        *ms++ = b;
        a = 0;
        b = 0;
        m = 1;
      }
      else
        m <<= 1;
    }
    r.pattern.setlength(rl);
    r.mask.setlength(rl);
  }
  return r;
}

ostream& operator << (ostream& s, BitPattern& x)
{
  return s << BitPatterntoa(x);
}

