/* 
 * predicate.c
 */

#include <stdio.h>
#include "rtl.h"
#include "insn-config.h"
#include "tree.h"
#include "predicate.h"

pred_list all_loop_predicates;

rtx get_array_rtx ();
 
predicate
predicate_create (type, variable, arr_ref, data, old_pred, start_insn, 
		  end_insn, load_insn, loop_init_value, loop_depth,
		  lower_bound, coefficient, iterator_rtx)
     int type;
     tree variable;
     tree arr_ref;
     int data;
     predicate old_pred;
     rtx start_insn;
     rtx end_insn;
     rtx load_insn;
     tree loop_init_value;
     int loop_depth;
     int lower_bound;
     float coefficient;
     rtx iterator_rtx;
{
  predicate p;
  tree arg0;
  tree arg1;

  p = (predicate) malloc (sizeof (struct predicate_def));
  p->type = type;
  p->variable = variable;
  p->arr_ref = get_array_rtx (arr_ref);

  switch (type)
    {
    case SPATIAL_PREDICATE:
      if (!data)
	fprintf (stderr, "ERROR: stride == 0 in predicate_create\n");
      p->info.stride = data;
      break;
    case TEMPORAL_PREDICATE:
      p->info.init = data;
      break;
    default:
      fprintf (stderr, "ERROR: Unknown predicate type %d\n", type);
    }
  p->start_insn = start_insn;
  p->end_insn = end_insn;
  p->load_insn = load_insn;
  /* Convert loop_init_value to a integer */
  if (loop_init_value)
    {
      if ((TREE_CODE (loop_init_value) == MODIFY_EXPR) &&
	  (TREE_CODE (TREE_OPERAND (loop_init_value, 1)) == INTEGER_CST))
	{
	  p->loop_init_value = 
	    TREE_INT_CST_LOW (TREE_OPERAND (loop_init_value, 1));
	} else {
	  p->loop_init_value = -1;
	}
    } else {
      p->loop_init_value = -1;
    }
  p->offset = ref_offset (arr_ref);
  p->loop_depth = loop_depth;
  p->next = old_pred;
  p->lower_bound = lower_bound;
  p->coefficient = coefficient;
  p->iterator_rtx = iterator_rtx;
  if (GET_CODE (iterator_rtx) == REG)
    {
      p->iterator_reg = REGNO (iterator_rtx);
    } else {
      p->iterator_reg = 0;
    }
  return p;
}

void
predicate_delete (p)
     predicate p;
{
  predicate curr;
  predicate old;

  for (curr = p; curr; )
    {
      old = curr;
      curr = curr->next;
      free (old);
    }
}

pred_list
pred_list_create (lp, note, next)
     loop_predicates lp;
     rtx note;
     pred_list next;
{
  pred_list pl;

  pl = (pred_list) malloc (sizeof (struct pred_list_def));
  
  pl->loop_pred = lp;
  pl->loop_end_note = note;
  pl->next_loop = next;
  
  return pl;
}

/* Returns first predicate matching the loop end note */
loop_predicates
get_loop_pred (pl, loop_end_note)
     pred_list pl;
     rtx loop_end_note;
{
  pred_list curr;

  if (pl == NULL)
    return NULL;

  for (curr = pl; curr; curr = curr->next_loop)
    {
      if (curr->loop_end_note == loop_end_note)
	return curr->loop_pred;
    }

  if (curr == NULL)
    return NULL;
}


void debug_predicate (p)
     predicate p;
{
  predicate curr;

  for (curr = p; curr; curr = curr->next)
    { 
      fprintf (stderr, " (");
      switch (curr->type)
	{
	case SPATIAL_PREDICATE:
	  fprintf (stderr, "S ");
	  print_expr(stderr, curr->variable);
	  fprintf (stderr, " %d ", curr->info.stride);
	  break;
	case TEMPORAL_PREDICATE:
	  fprintf (stderr, "T ");
	  print_expr(stderr, curr->variable);
	  fprintf (stderr, " %d ", curr->info.init);
	  break;
	}
      if (curr->start_insn)
	fprintf (stderr, " (%d to ", INSN_UID (curr->start_insn));
      else 
	fprintf (stderr, " (NULL to ");

      if (curr->end_insn)
	fprintf (stderr, " %d)", INSN_UID (curr->end_insn));
      else 
	fprintf (stderr, "NULL)");

/*      print_arr_ref (stderr, curr->arr_ref);*/
      if (curr->load_insn)
	fprintf (stderr, " load %d ", INSN_UID (curr->load_insn));	
      fprintf (stderr, " init %d", curr->loop_init_value);
      fprintf (stderr, " offset %d", curr->offset);
      if (curr->type == SPATIAL_PREDICATE)
	fprintf (stderr, " coefficient %.2f", curr->coefficient);
      /* coefficient doesn't matter for temporal predicates */
      fprintf (stderr, " depth %d ) ", curr->loop_depth);
      fprintf (stderr, "\n\t");
    }
  fprintf (stderr, "\n");
}


void predicate_print (f, p)
     FILE *f;
     predicate p;
{
  predicate curr;

  for (curr = p; curr; curr = curr->next)
    { 
      fprintf (f, " (");
      switch (curr->type)
	{
	case SPATIAL_PREDICATE:
	  fprintf (f, "S ");
	  print_expr(f, curr->variable);
	  fprintf (f, " %d ", curr->info.stride);
	  break;
	case TEMPORAL_PREDICATE:
	  fprintf (f, "T ");
	  print_expr(f, curr->variable);
	  fprintf (f, " %d ", curr->info.init);
	  break;
	}
      if (curr->start_insn)
	fprintf (f, " (%d to ", INSN_UID (curr->start_insn));
      else 
	fprintf (f, " (NULL to ");

      if (curr->end_insn)
	fprintf (f, " %d) ", INSN_UID (curr->end_insn));
      else 
	fprintf (f, "NULL) ");

/*      print_arr_ref (f, curr->arr_ref);*/
      if (curr->load_insn)
	fprintf (f, " load %d ", INSN_UID (curr->load_insn));
      fprintf (f, " init %d", curr->loop_init_value);
      fprintf (f, " offset %d", curr->offset);
      if (curr->type == SPATIAL_PREDICATE)
	fprintf (f, " coefficient %.2f", curr->coefficient);
      /* coefficient doesn't matter for temporal predicates */
      fprintf (f, " depth %d ) ", curr->loop_depth);
      fprintf (f, "\n\t");
    }
  fprintf (f, "\n");
}

void pred_list_print (f, pl)
     FILE *f;
     pred_list pl;
{
  loop_predicates lp;
  pred_list curr;
  for (curr = pl; curr; curr = curr->next_loop)
    {
      lp = curr->loop_pred;
      fprintf (f, "-- %d ", INSN_UID (curr->loop_end_note));
      loop_predicates_print (f, lp);
    }
}

void debug_pred_list (pl)
     pred_list pl;
{
  loop_predicates lp;
  pred_list curr;
  for (curr = pl; curr; curr = curr->next_loop)
    {
      lp = curr->loop_pred;
      fprintf (stderr, "-- %d ", INSN_UID (curr->loop_end_note));
      debug_loop_predicates (lp);
    }
}

void pred_list_delete (pl)
     pred_list pl;
{
  loop_predicates lp;
  pred_list curr;
  for (curr = pl; curr; curr = curr->next_loop)
    {
      lp = curr->loop_pred;
      loop_predicates_delete (lp);
    }
}


void 
predicate_set_end_insn (curr_temporal_pred, curr_spatial_pred, end_insn)
     predicate curr_temporal_pred;
     predicate curr_spatial_pred;
     rtx end_insn;
{
  if (curr_spatial_pred)
    curr_spatial_pred->end_insn = end_insn;
  if (curr_temporal_pred)
    curr_temporal_pred->end_insn = end_insn;
}

predicate 
predicate_copy (p, new_next)
     predicate p;
     predicate new_next;
{
  predicate new_p;

  new_p = (predicate) malloc (sizeof (struct predicate_def));
  new_p->type = p->type;
  new_p->variable = p->variable;
  new_p->arr_ref = p->arr_ref;
  switch (p->type)
    {
    case SPATIAL_PREDICATE:
      new_p->info.stride = p->info.stride;
      break;
    case TEMPORAL_PREDICATE:
      new_p->info.init = p->info.init;
      break;
    default:
      fprintf (stderr, "ERROR: Unknown predicate type %d\n", p->type);
    }
  new_p->start_insn = p->start_insn;
  new_p->end_insn = p->end_insn;
  new_p->load_insn = p->load_insn;
  new_p->next = new_next;
  new_p->offset = p->offset;
  new_p->loop_init_value = p->loop_init_value;
  new_p->loop_depth = p->loop_depth;
  new_p->lower_bound = p->lower_bound;
  new_p->coefficient = p->coefficient;
  new_p->iterator_rtx = p->iterator_rtx;
  new_p->iterator_reg = p->iterator_reg;

  return new_p;
}

loop_predicates
loop_predicates_create (p, next)
     predicate p;
     loop_predicates next;
{
  loop_predicates pl;

  pl = (loop_predicates) malloc (sizeof (struct loop_predicates_def));
  
  pl->pred = p;
  pl->next_arr_ref = next;
  
  return pl;
}

void loop_predicates_print (f, pl)
     FILE *f;
     loop_predicates pl;
{
  predicate p;
  loop_predicates curr;
  int i;

  for (curr = pl, i = 0; curr; curr = curr->next_arr_ref, i++)
    {
      p = curr->pred;
      fprintf (f, "Ref %d ", i);
      predicate_print (f, p);
    }
}

void debug_loop_predicates (pl)
     loop_predicates pl;
{
  predicate p;
  loop_predicates curr;
  int i;

  for (curr = pl, i = 0; curr; curr = curr->next_arr_ref, i++)
    {
      p = curr->pred;
      fprintf (stderr, "Ref %d ", i);
      debug_predicate (p);
    }
}

void loop_predicates_delete (pl)
     loop_predicates pl;
{
  predicate p;
  loop_predicates curr;

  for (curr = pl; curr; curr = curr->next_arr_ref)
    {
      p = curr->pred;
      predicate_delete (p);
    }
}


void
update_predicate_load_insns (pl)
     pred_list pl;
{
  loop_predicates lp, curr_lp;
  pred_list curr;
  predicate p;

  for (curr = pl; curr; curr = curr->next_loop)
    {
      lp = curr->loop_pred;

      for (curr_lp = lp; curr_lp; curr_lp = curr_lp->next_arr_ref)
	{
	  p = curr_lp->pred;
	  find_load_for_predicate (p);
	}
    }
}

rtx
get_array_rtx (arr_ref)
     tree arr_ref;
{
  rtx temp;

  switch (TREE_CODE (arr_ref))
    {
    case ARRAY_REF:
    case COMPONENT_REF:
      return get_array_rtx (TREE_OPERAND (arr_ref, 0));
      break;
    case VAR_DECL:
      temp = DECL_RTL (arr_ref);
      if (GET_CODE (temp) == MEM)
	temp = XEXP (temp, 0);
      return temp;
      break;
    }
}
