/* Breadth-first and depth-first routines for
   searching multiple-inheritance lattice for GNU C++.
   Copyright (C) 1987 Free Software Foundation, Inc.
   Contributed by Michael Tiemann (tiemann@mcc.com)

   This file is part of GNU CC.
   
   GNU CC is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.  No author or distributor
   accepts responsibility to anyone for the consequences of using it
   or for whether it serves any particular purpose or works at all,
   unless he says so in writing.  Refer to the GNU CC General Public
   License for full details.
   
   Everyone is granted permission to copy, modify and redistribute
   GNU CC, but only under the conditions described in the
   GNU CC General Public License.   A copy of this license is
   supposed to have been given to you along with GNU CC so you
   can know your rights and responsibilities.  It should be in a
   file named COPYING.  Among other things, the copyright notice
   and this notice must be preserved on all copies.  */


/* High-level class interface. */

#include "config.h"
#include "tree.h"
#include "cplus-tree.h"
#include "obstack.h"
#include "flags.h"
#include "assert.h"

#define NULL 0

#define obstack_chunk_alloc xmalloc
#define obstack_chunk_free free

extern int xmalloc ();
extern void free ();

void init_search ();

/* Stack of places to restore the search obstack back to.  */
   
struct search_level
{
  /* Pointer back to previous such level.  */
  struct search_level *prev;
  /* First object allocated within this level.  */
  char *base;
  /* First place we start putting data.  */
  tree *first;
};

/* Obstack used for remembering decision points of breadth-first.  */
static struct obstack search_obstack;
static struct search_level *search_stack;

struct type_level
{
  /* Pointer back to previous such level.  */
  struct type_level *prev;
  /* First object allocated within this level.  */
  char *base;
  /* First object allocated in obstack of entries.  */
  char *entries;
  /* Number of types in this obstack.  */
  int len;
  /* Type being memoized.  */
  tree type;

  /* First place we start putting data.  */
  tree *first;
};

/* Obstack used for memoizing member and member function lookup.  */

static struct obstack type_obstack, type_obstack_entries;
static struct type_level *type_stack;

/* Make things that look like tree nodes, but allocate them
   on type_obstack_entries.  */
static int my_tree_node_counter;
static tree my_tree_cons (), my_build_string ();

extern int flag_memoize_lookups, flag_save_memoized_contexts;

/* Variables for gathering statistics.  */
static int my_memoized_entry_counter;
static int memoized_fast_finds[2], memoized_adds[2], memoized_fast_rejects[2];
static int memoized_fields_searched[2];
static int n_fields_searched;
static int n_calls_lookup_field, n_calls_lookup_field_1;
static int n_calls_lookup_fnfields, n_calls_lookup_fnfields_1;
static int n_outer_fields_searched;
static int n_contexts_saved;

/* Local variables to help save memoization contexts.  */
static tree prev_type_memoized;
static struct type_level *prev_type_stack;

static tree lookup_fields_1 (), lookup_fnfields_1 ();

/* Allocate a level of searching.  */
static struct search_level *
push_search_level (stack, obstack)
     struct search_level *stack;
     struct obstack *obstack;
{
  struct search_level tem;

  tem.prev = stack;
  tem.base = (char *) obstack_base (obstack);
  stack = (struct search_level *) obstack_next_free (obstack);
  obstack_grow (obstack, &tem, sizeof (tem));
  stack->first = (tree *) obstack_next_free (obstack);
  return stack;
}

/* Discard a level of search allocation.  */

static struct search_level *
pop_search_level (stack, obstack)
     struct search_level *stack;
     struct obstack *obstack;
{
  struct search_level *tem = stack;
  stack = tem->prev;
  obstack_free (obstack, tem);
  return stack;
}

/* Allocate a level of type memoziation context.  */
static struct type_level *
push_type_level (stack, obstack)
     struct type_level *stack;
     struct obstack *obstack;
{
  struct type_level tem;

  tem.prev = stack;
  tem.base = (char *) obstack_base (obstack);
  tem.entries = (char *) obstack_next_free (&type_obstack_entries);
  tem.len = 0;
  tem.type = NULL_TREE;
  stack = (struct type_level *) obstack_next_free (obstack);
  obstack_grow (obstack, &tem, sizeof (tem));
  stack->first = (tree *) obstack_next_free (obstack);
  return stack;
}

/* Discard a level of type memoziation context.  */

static struct type_level *
pop_type_level (stack, obstack)
     struct type_level *stack;
     struct obstack *obstack;
{
  struct type_level *tem = stack;
  stack = tem->prev;
  obstack_free (&type_obstack_entries, tem->entries);
  obstack_free (obstack, tem);
  return stack;
}

/* Make something that looks like a TREE_LIST, but
   do it on the type_obstack_entries obstack.  */
static tree
my_tree_cons (purpose, value, chain)
     tree purpose, value, chain;
{
  tree p = obstack_alloc (&type_obstack_entries, sizeof (struct tree_list));
  bzero (&p->common, sizeof (p->common));
  TREE_UID (p) = ++my_tree_node_counter;
  TREE_CODE (p) = TREE_LIST;
  TREE_PURPOSE (p) = purpose;
  TREE_VALUE (p) = value;
  TREE_CHAIN (p) = chain;
  return p;
}

static tree
my_build_string (str)
     char *str;
{
  tree p = obstack_alloc (&type_obstack_entries, sizeof (struct tree_string));
  bzero (&p->common, sizeof (p->common));
  TREE_UID (p) = ++my_tree_node_counter;
  TREE_CODE (p) = STRING_CST;
  TREE_STRING_POINTER (p) = str;
  TREE_STRING_LENGTH (p) = strlen (str);
  return p;
}

/* Memoizing machinery to make searches for multiple inheritance
   reasonably efficient.  */
#define MEMOIZE_HASHSIZE 4
typedef struct memoized_entry
{
  struct memoized_entry *chain;
  int uid;
  tree data_members[MEMOIZE_HASHSIZE];
  tree function_members[MEMOIZE_HASHSIZE];
} *ME;

#define MEMOIZED_CHAIN(ENTRY) (((ME)ENTRY)->chain)
#define MEMOIZED_UID(ENTRY) (((ME)ENTRY)->uid)
#define MEMOIZED_FIELDS(ENTRY,INDEX) (((ME)ENTRY)->data_members[INDEX])
#define MEMOIZED_FNFIELDS(ENTRY,INDEX) (((ME)ENTRY)->function_members[INDEX])
#define MEMOIZED_HASH_FN(NODE) ((((int)NODE) >> 3)&(MEMOIZE_HASHSIZE - 1))

static struct memoized_entry *
my_new_memoized_entry (chain)
     struct memoized_entry *chain;
{
  struct memoized_entry *p =
    (struct memoized_entry *)obstack_alloc (&type_obstack_entries,
					    sizeof (struct memoized_entry));
  bzero (p, sizeof (struct memoized_entry));
  MEMOIZED_CHAIN (p) = chain;
  MEMOIZED_UID (p) = ++my_memoized_entry_counter;
  return p;
}

/* When a new function or class context is entered, we build
   a table of types which have been searched for members.
   The table is an array (obstack) of types.  When a type is
   entered into the obstack, is CLASSTYPE_MTABLE_ENTRY
   field is set to point to a new record, of type struct memoized_entry.

   The slots for the data members are arrays of tree nodes.
   These tree nodes are lists, with the TREE_PURPOSE
   of this list the known member name, and the TREE_VALUE
   as the FIELD_DECL for the member.

   For member functions, the TREE_PURPOSE is again the
   name of the member functions for that class,
   and the TREE_VALUE of the list is a pair
   whose TREE_PURPOSE is the list of member functions of this name,
   and whose TREE_VALUE is a list of known argument lists this
   member function has been called with.

   If the current argument list matches the TREE_TYPE of an
   element of the argument-list list, then then the TREE_VALUE
   of that element is the member which should actually be used,
   and the TREE_PURPOSE is an error message to print.

   For data members (and function members from base classes),
   if the TREE_TYPE of the entry is non-NULL, it contains a visibility
   error message.  */

/* Tell search machinery that we are entering a new context, and
   to update tables appropriately.

   TYPE is the type of the context we are entering, which can
   be NULL_TREE if we are not in a class's scope.

   USE_OLD, if nonzero tries to use previous context.  */
push_memoized_context (type, use_old)
     tree type;
     int use_old;
{
  int len;
  tree *tem;

  if (prev_type_stack)
    {
      if (use_old && prev_type_memoized == type)
	{
	  n_contexts_saved++;
	  type_stack = prev_type_stack;
	  prev_type_stack = 0;
	  len = type_stack->len;
	  tem = &type_stack->first[0];
	  while (len--)
	    {
	      CLASSTYPE_MTABLE_ENTRY (tem[len*2]) = tem[len*2+1];
	    }
	  return;
	}
      /* Need to pop old stack here.  */

      type_stack = pop_type_level (prev_type_stack, &type_obstack);
      prev_type_memoized = 0;
      prev_type_stack = 0;
    }
  if (type_stack)
    {
      len = type_stack->len;
      tem = &type_stack->first[0];
      while (len--)
	{
	  CLASSTYPE_MTABLE_ENTRY (tem[len*2])
	    = my_new_memoized_entry (CLASSTYPE_MTABLE_ENTRY (tem[len*2]));
	}
    }
  if (flag_memoize_lookups)
    {
      type_stack = push_type_level (type_stack, &type_obstack);
      type_stack->type = type;
    }
}

/* Tell search machinery that we have left a context.
   We do not currently save these contexts for later use.
   If we wanted to, we could not use pop_search_level, since
   poping that level allows the data we have collected to
   be clobbered; a stack of obstacks would be needed.  */
pop_memoized_context (use_old)
     int use_old;
{
  int i, len;
  tree *tem;

  assert (prev_type_stack == 0);

  if (type_stack)
    {
      if (! flag_save_memoized_contexts)
	use_old = 0;

      len = type_stack->len;
      tem = &type_stack->first[0];
      if (use_old)
	{
	  while (len--)
	    {
	      tem[len*2+1] = (tree)CLASSTYPE_MTABLE_ENTRY (tem[len*2]);
	    }
	  prev_type_stack = type_stack;
	  prev_type_memoized = type_stack->type;
	}

      len = type_stack->len;
      while (len--)
	{
	  CLASSTYPE_MTABLE_ENTRY (tem[len*2])
	    = MEMOIZED_CHAIN (CLASSTYPE_MTABLE_ENTRY (tem[len*2]));
	}

      if (! use_old)
	{
	  type_stack = pop_type_level (type_stack, &type_obstack);
	}
    }
}

/* Some simple list processing predicates.  */

/* Check whether TYPE is immediately derived from PARENT.
   Return actual base information if so.  Otherwise, return 0.  */
tree
get_base_type_1 (parent, type)
     register tree parent, type;
{
  register int i;

  parent = CLASSTYPE_MAIN_VARIANT (parent);
  type = CLASSTYPE_MAIN_VARIANT (type);

  for (i = 1; i < CLASSTYPE_N_BASECLASSES (type); i++)
    if (CLASSTYPE_MAIN_VARIANT (CLASSTYPE_BASECLASS (type, i)) == parent)
      return CLASSTYPE_BASECLASS (type, i);

  return 0;
}

/* Check whether TYPE is derived from PARENT.
   Return the actual base information if so.  Otherwise return 0.  */
tree
get_base_type (parent, type)
     register tree parent, type;
{
  int head = 0, tail = 0;
  tree rval;

  search_stack = push_search_level (search_stack, &search_obstack);

  parent = CLASSTYPE_MAIN_VARIANT (parent);
  type = CLASSTYPE_MAIN_VARIANT (type);

  while (1)
    {
      int i, n_baselinks = CLASSTYPE_N_BASECLASSES (type);

      /* Process and/or queue base types.  */
      for (i = 1; i <= n_baselinks; i++)
	if (CLASSTYPE_MARKED (CLASSTYPE_BASECLASS (type, i)) == 0)
	  {
	    CLASSTYPE_MARKED (CLASSTYPE_BASECLASS (type, i)) = 1;
	    obstack_grow (&search_obstack, &CLASSTYPE_BASECLASS (type, i), sizeof (tree *));
	    tail++;
	  }

      /* Process head of queue, if one exists.  */
      if (head >= tail)
	{
	  rval = 0;
	  break;
	}


      type = search_stack->first[head++];
      if (CLASSTYPE_MAIN_VARIANT (type) == parent)
	{
	  rval = type;
	  break;
	}
    }
  {
    tree *tp = search_stack->first;
    tree *search_tail = tp + tail;

    while (tp < search_tail)
      {
	CLASSTYPE_MARKED (*tp++) = 0;
      }
  }
  search_stack = pop_search_level (search_stack, &search_obstack);
  return rval;
}

/* Search for a member with name NAME in a multiple inheritance lattice
   specified by TYPE.  If it does not exist, return NULL_TREE.
   If the member is ambiguously referenced, return `error_mark_node'.
   Otherwise, return the FIELD_DECL.  */

/* Do a 1-level search for NAME as a member of TYPE.  The caller
   must figure out whether it has a visible path to this field.
   (Since it is only one level, this is reasonable.)  */
static tree
lookup_field_1 (type, name)
     tree type, name;
{
  register tree field = TYPE_FIELDS (type);

  n_calls_lookup_field_1++;
  while (field)
    {
      n_fields_searched++;
      if (DECL_NAME (field) == name)
	break;
      field = TREE_CHAIN (field);
    }
  return field;
}

/* Compute the visibility of FIELD.  This is done by computing
   the visibility available to each type in BASETYPES (which comes
   as a list of [via_public/basetype] in reverse order, namely base
   class before derived class).  The first one which defines a
   visibility defines the visibility for the field.  Otherwise, the
   visibility of the field is that which occurs normally.

   Uses global variables CURRENT_CLASS_TYPE and
   CURRENT_FUNCTION_DECL to use friend relationships
   if necessary.
   VISIBILITY may come in as something other than VISIBILITY_DEFAULT.
   In that case, unless FIELD has special visibility for BASETYPE
   or BASETYPE is a friend of CURRENT_CLASS_TYPE, it may override
   what FIELD is willing to offer.

   This will be static when lookup_fnfield comes into this file.  */

enum visibility_type
compute_visibility (basetypes, field)
     tree basetypes, field;
{
  enum visibility_type visibility = visibility_public;
  tree types, type;

  /* Virtual function tables are never private.
     But we should know that we are looking for this,
     and not even try to hide it.  */
  assert (VFIELD_NAME_P (DECL_NAME (field)) == 0);

  /* Make these special cases fast.  */
  /* Member function manipulating its own members.  */
  if (current_class_type == DECL_CONTEXT (field))
    return visibility_public;
  /* Member found immediately within object.  */
  if (TREE_CHAIN (basetypes) == NULL_TREE)
    {
      /* At object's top level, public members are public.  */
      if (TREE_PUBLIC (field))
	return visibility_public;

      /* Friend function manipulating members it gets (for being a friend).  */
      if (is_friend (DECL_CONTEXT (field), current_function_decl))
	return visibility_public;

      /* Inner than that, without special visibility,

	   protected members are ok if current_class_type is derived
	   from the type of the object.

	   private members are not ok.  */
      if (current_class_type && DECL_VISIBILITY (field) == NULL_TREE)
	{
	  if (TREE_PRIVATE (field))
	    return visibility_private;

	  if (TREE_PROTECTED (field))
	    {
	      type = get_base_type (DECL_CONTEXT (field), current_class_type);
	      if (type)
		return visibility_public;
	      return visibility_protected;
	    }
	}
    }
  else
    {
      /* Friend function manipulating members it gets (for being a friend).  */
      if (is_friend (DECL_CONTEXT (field), current_function_decl))
	return visibility_public;

      /* must reverse more than one element */
      basetypes = nreverse (basetypes);
    }

  types = basetypes;

  while (types)
    {
      tree member;
      type = CLASSTYPE_MAIN_VARIANT (TREE_VALUE (types));

      member = purpose_member (type, DECL_VISIBILITY (field));
      if (member)
	{
	  nreverse (basetypes);
	  visibility = (enum visibility_type)TREE_VALUE (member);
	  if (visibility == visibility_public)
	    return visibility;
	  if (is_friend (type, current_function_decl))
	    return visibility_public;
	  if (visibility == visibility_protected
	      && current_class_type
	      && get_base_type (DECL_CONTEXT (field), current_class_type))
	    return visibility_public;
	  return visibility;
	}

      /* Friends inherit the visibility of the class they inherit from.  */
      if (is_friend (type, current_function_decl))
	{
	  if (TREE_PROTECTED (field))
	    return visibility_public;
	  if (TREE_PRIVATE (field))
	    if (type == DECL_CONTEXT (field))
	      return visibility_public;
	    else
	      /* may be a friend of a deeper base class */ ;
	  else if (visibility == visibility_public)
	    return visibility_public;
	  else
	    /* may be a friend of a deeper base class */ ;
	}

      types = TREE_CHAIN (types);
      /* If the next type was not VIA_PUBLIC, then fields of all
	 remaining class past that one are private.  */
      if (types && TREE_PURPOSE (types) == NULL_TREE)
	visibility = visibility_private;
    }

  /* No special visibilities apply.  Use normal rules.
     No assignment needed for BASETYPEs here from the nreverse.
     This is because we use it only for information about the
     path to the base.  The code earlier dealt with what
     happens when we are at the base level.  */
  basetypes = nreverse (basetypes);

  if (visibility == visibility_public)
    if (TREE_PRIVATE (field))
      return visibility_private;
    else if (TREE_PROTECTED (field))
      {
	/* Used to check if the current class type was derived from
	   the type that contains the field.  This is wrong for
	   multiple inheritance because is gives one class reference
	   to protected members via another classes protected path.
	   I.e., if A; B1 : A; B2 : A;  Then B1 and B2 can access
	   their own members which are protected in A, but not
	   those same members in one another.  */
#if 0
	if (current_class_type
	    && get_base_type (DECL_CONTEXT (field), current_class_type))
#else
	if (current_class_type && value_member (current_class_type, basetypes))
#endif
	  return visibility_public;
	else
	  return visibility_protected;
      }
    else
      return visibility_public;

  if (visibility == visibility_private
      && current_class_type != NULL_TREE)
    {
      int i, n_baselinks = CLASSTYPE_N_BASECLASSES (current_class_type);

      /* See if the field isn't protected.  */
      if (TREE_PROTECTED (field))
	{
#if 0
	  if (get_base_type (type, current_class_type))
#else
	  if (value_member (current_class_type, basetypes))
#endif
	    return visibility_public;
	  else
	    return visibility_protected;
	}

      /* See if the field isn't a public member of
	 a private base class.  */

      type = DECL_CONTEXT (field);
      if (! TREE_PRIVATE (field))
	for (i = 1; i <= n_baselinks; i++)
	  if (CLASSTYPE_MAIN_VARIANT (CLASSTYPE_BASECLASS (current_class_type, i)) == type)
	    return visibility_public;
    }

  return visibility;
}

/* Make an entry in the memoized table for type TYPE
   that the entry for NAME is FIELD.  */

tree
make_memoized_table_entry (type, name, function_p)
     tree type, name;
     int function_p;     
{
  int index = MEMOIZED_HASH_FN (name);
  tree entry, *prev_entry;

  memoized_adds[function_p] += 1;
  if (CLASSTYPE_MTABLE_ENTRY (type) == NULL_TREE)
    {
      obstack_grow (&type_obstack, &type, sizeof (tree *));
      obstack_blank (&type_obstack, sizeof (struct memoized_entry *));
      CLASSTYPE_MTABLE_ENTRY (type) = my_new_memoized_entry (0);
      type_stack->len++;
    }
  if (function_p)
    prev_entry = &MEMOIZED_FNFIELDS (CLASSTYPE_MTABLE_ENTRY (type), index);
  else
    prev_entry = &MEMOIZED_FIELDS (CLASSTYPE_MTABLE_ENTRY (type), index);

  entry = my_tree_cons (name, 0, *prev_entry);
  *prev_entry = entry;

  /* Don't know the error message to give yet.  */
  TREE_TYPE (entry) = error_mark_node;

  return entry;
}

tree
lookup_field (basetype, name, protect)
     register tree basetype, name;
     int protect;
{
  int head = 0, tail = 0;
  tree type, rval;
  tree basetypes = build_tree_list (NULL_TREE, basetype);
  enum visibility_type this_v = visibility_default;
  tree entry;

  /* Things for memoization.  */
  char *errstr = 0;

  /* Set this to nonzero if we don't know how to compute
     accurate error messages for visibility.  */
  int memoization_too_hard = 0;
  int index = MEMOIZED_HASH_FN (name);

  if (CLASSTYPE_MTABLE_ENTRY (basetype))
    {
      tree tem = MEMOIZED_FIELDS (CLASSTYPE_MTABLE_ENTRY (basetype), index);

      while (tem && TREE_PURPOSE (tem) != name)
	{
	  memoized_fields_searched[0]++;
	  tem = TREE_CHAIN (tem);
	}
      if (tem)
	{
	  if (protect && TREE_TYPE (tem))
	    {
	      error (TREE_STRING_POINTER (TREE_TYPE (tem)),
		     IDENTIFIER_POINTER (name),
		     TYPE_NAME_STRING (DECL_CONTEXT (TREE_VALUE (tem))));
	      return error_mark_node;
	    }
	  if (TREE_VALUE (tem) == NULL_TREE)
	    memoized_fast_rejects[0] += 1;
	  else
	    memoized_fast_finds[0] += 1;
	  return TREE_VALUE (tem);
	}
    }

  n_calls_lookup_field++;
  if (type_stack)
    entry = make_memoized_table_entry (basetype, name, 0);
  else
    entry = 0;

  rval = lookup_field_1 (basetype, name);

  if (rval)
    {
      if (flag_memoize_lookups || protect)
	{
	  if (TREE_PRIVATE (rval) | TREE_PROTECTED (rval))
	    this_v = compute_visibility (basetypes, rval);
	  if (this_v == visibility_private)
	    errstr = "member `%s' is a private member of class `%s'";
	  else if (this_v == visibility_protected)
	    errstr = "member `%s' is a protected member of class `%s'";
	}

      if (entry)
	{
	  if (errstr)
	    {
	      /* This depends on behavior of lookup_field_1!  */
	      tree error_string = my_build_string (errstr);
	      TREE_TYPE (entry) = error_string;
	    }
	  else
	    {
	      /* Let entry know there is no problem with this access.  */
	      TREE_TYPE (entry) = NULL_TREE;
	    }
	  TREE_VALUE (entry) = rval;
	}

      if (errstr && protect)
	{
	  error (errstr, IDENTIFIER_POINTER (name), TYPE_NAME_STRING (basetype));
	  return error_mark_node;
	}
      return rval;
    }

  type = CLASSTYPE_MAIN_VARIANT (basetype);

  search_stack = push_search_level (search_stack, &search_obstack);

  while (1)
    {
      int i, n_baselinks = CLASSTYPE_N_BASECLASSES (type);

      /* Process and/or queue base types.  */
      for (i = 1; i <= n_baselinks; i++)
	{
	  if (CLASSTYPE_MARKED2 (CLASSTYPE_BASECLASS (type, i)) == 0)
	    {
	      tree btypes = basetypes;

	      CLASSTYPE_MARKED2 (CLASSTYPE_BASECLASS (type, i)) = 1;
	      obstack_grow (&search_obstack, &CLASSTYPE_BASECLASS (type, i), sizeof (tree *));
	      btypes = tree_cons (CLASSTYPE_VIA_PUBLIC (type, i),
				  CLASSTYPE_BASECLASS (type, i), basetypes);
	      obstack_grow (&search_obstack, &btypes, sizeof (tree *));
	      tail += 2;
	    }
	}

      /* Process head of queue, if one exists.  */
      if (head >= tail)
	break;

      type = search_stack->first[head++];
      basetypes = search_stack->first[head++];

      /* See if we can find NAME in TYPE.  If RVAL is nonzero,
	 and we do find NAME in TYPE, verify that such a second
	 sighting is in fact legal.  */

      if (rval)
	{
	  /* Just another way of finding the same member.  */
	  if (DECL_CONTEXT (rval) == type)
	    {
	      enum visibility_type new_v
		= compute_visibility (basetypes, rval);
	      if (this_v != new_v)
		errstr = "conflicting visibilities to member `%s'";
	    }
	  /* Same baseclass, different places in the lattice.  */
	  else if (CLASSTYPE_MAIN_VARIANT (DECL_CONTEXT (rval))
		   == CLASSTYPE_MAIN_VARIANT (type))
	    errstr = "member `%s' belongs to distinct base classes `%s'";
	  else
	    {
	      tree nval = lookup_field_1 (type, name);

	      if (nval && get_base_type (type, DECL_CONTEXT (rval)) == 0)
		{
		  /* We found it in other than a baseclass of RVAL's.  */
		  errstr = "request for member `%s' is ambiguous";
		}
	    }
	  if (errstr && entry)
	    {
	      tree error_string = my_build_string (errstr);
	      TREE_TYPE (entry) = error_string;
	    }
	  if (errstr && protect)
	    break;
	}
      else
	{
	  rval = lookup_field_1 (type, name);
	  if (rval)
	    {
	      if (entry || protect)
		this_v = compute_visibility (basetypes, rval);
	      if (entry)
		TREE_VALUE (entry) = rval;
	    }
	}
    }
  {
    tree *tp = search_stack->first;
    tree *search_tail = tp + tail;

    /* If this FIELD_DECL defines its own visibility, deal with that.

       @@ Arguments to compute_visibility here are wrong!  */
    if (rval && errstr == 0
	&& DECL_VISIBILITY (rval)
	&& (protect || entry))
      {
	assert (0);
	while (tp < search_tail)
	  {
	    /* If is possible for one of the derived types on the
	       path to have defined special visibility for this
	       field.  Look for such declarations and report an
	       error if a conflict is found.  */
	    enum visibility_type new_v;

	    if (this_v != visibility_default)
	      new_v = compute_visibility (tp[1], tp[0], rval);
	    if (this_v != visibility_default && new_v != this_v)
	      {
		errstr = "conflicting visibilities to member `%s'";
		this_v = visibility_default;
	      }
	    CLASSTYPE_MARKED2 (*tp++) = 0;
	    tp++;
	  }
      }
    else
      {
	while (tp < search_tail)
	  {
	    CLASSTYPE_MARKED2 (*tp++) = 0;
	    tp++;
	  }
      }
  }
  search_stack = pop_search_level (search_stack, &search_obstack);

  if (errstr == 0)
    {
      if (this_v == visibility_private)
	errstr = TREE_PRIVATE (rval) ? "member `%s' is private" : "member `%s' is from private base class";
      else if (this_v == visibility_protected)
	errstr = "member `%s' is protected";
    }

  if (entry)
    {
      if (errstr)
	{
	  tree error_string = my_build_string (errstr);
	  /* Save error message with entry.  */
	  TREE_TYPE (entry) = error_string;
	}
      else
	{
	  /* Mark entry as having no error string.  */
	  TREE_TYPE (entry) = NULL_TREE;
	}
    }

  if (errstr && protect)
    {
      error (errstr, IDENTIFIER_POINTER (name), TYPE_NAME_STRING (type));
      rval = error_mark_node;
    }
  return rval;
}

/* TYPE is a class type. Return the list of fields with name
   NAME, or NULL_TREE is no such field exists.  */
static tree
lookup_fnfields_1 (type, name)
     tree type, name;
{
  register tree field = TYPE_FN_FIELDS (type);

  n_calls_lookup_fnfields_1++;
  while (field)
    {
      n_outer_fields_searched++;	
      if (TREE_PURPOSE (field) == name)
	break;
      field = TREE_CHAIN (field);
    }
  return field;
}

/* Given a list of member functions FIELDS (which are implicitly
   named TREE_PURPOSE (FIELDS), and come from base type
   DECL_CONTEXT (TREE_VALUE (FIELDS))), attempt to find the
   actual method which can accept (using conversions) PARMS.
   The types of PARMS are already computed in PARMTYPES.  */
tree
lookup_fnfield (fields, parms, parmtypes)
     tree fields, parms, parmtypes;
{
}

tree
lookup_fnfields (basetype, name)
     tree basetype, name;
{
  int head = 0, tail = 0;
  tree type, rval, rvals;
  tree basetypes = build_tree_list (NULL_TREE, basetype);
  enum visibility_type this_v = visibility_default;
  tree entry;

  /* For now, don't try this.  */
  int protect = 0;

  /* Things for memoization.  */
  char *errstr = 0;

  /* Set this to nonzero if we don't know how to compute
     accurate error messages for visibility.  */
  int memoization_too_hard = 0;
  int index = MEMOIZED_HASH_FN (name);

  if (CLASSTYPE_MTABLE_ENTRY (basetype))
    {
      tree tem = MEMOIZED_FNFIELDS (CLASSTYPE_MTABLE_ENTRY (basetype), index);

      while (tem && TREE_PURPOSE (tem) != name)
	{
	  memoized_fields_searched[0]++;
	  tem = TREE_CHAIN (tem);
	}
      if (tem)
	{
	  if (protect && TREE_TYPE (tem))
	    {
	      error (TREE_STRING_POINTER (TREE_TYPE (tem)),
		     IDENTIFIER_POINTER (name),
		     TYPE_NAME_STRING (DECL_CONTEXT (TREE_VALUE (tem))));
	      return error_mark_node;
	    }
	  if (TREE_VALUE (tem) == NULL_TREE)
	    memoized_fast_rejects[1] += 1;
	  else
	    memoized_fast_finds[1] += 1;
	  return TREE_VALUE (tem);
	}
    }

  n_calls_lookup_fnfields++;
  if (type_stack)
    entry = make_memoized_table_entry (basetype, name, 1);
  else
    entry = 0;

  rval = lookup_fnfields_1 (basetype, name);

  if (rval)
    {
      /* @@ Not computing visibility information yet.  */
      if (entry)
	{
	  TREE_VALUE (entry) = rval;
	  TREE_TYPE (entry) = NULL_TREE;
	}

      if (errstr && protect)
	{
	  error (errstr, IDENTIFIER_POINTER (name), TYPE_NAME_STRING (basetype));
	  return error_mark_node;
	}
      return rval;
    }

  type = CLASSTYPE_MAIN_VARIANT (basetype);

  search_stack = push_search_level (search_stack, &search_obstack);

  while (1)
    {
      int i, n_baselinks = CLASSTYPE_N_BASECLASSES (type);

      /* Process and/or queue base types.  */
      for (i = 1; i <= n_baselinks; i++)
	{
	  if (CLASSTYPE_MARKED2 (CLASSTYPE_BASECLASS (type, i)) == 0)
	    {
	      tree btypes = basetypes;

	      CLASSTYPE_MARKED2 (CLASSTYPE_BASECLASS (type, i)) = 1;
	      obstack_grow (&search_obstack, &CLASSTYPE_BASECLASS (type, i), sizeof (tree *));
	      btypes = tree_cons (CLASSTYPE_VIA_PUBLIC (type, i),
				  CLASSTYPE_BASECLASS (type, i), basetypes);
	      obstack_grow (&search_obstack, &btypes, sizeof (tree *));
	      tail += 2;
	    }
	}

      /* Process head of queue, if one exists.  */
      if (head >= tail)
	break;

      type = search_stack->first[head++];
      basetypes = search_stack->first[head++];

      /* See if we can find NAME in TYPE.  If RVAL is nonzero,
	 and we do find NAME in TYPE, verify that such a second
	 sighting is in fact legal.  */

      if (rval)
	{
	  /* Just another way of finding the same member.  */
	  if (DECL_CONTEXT (TREE_VALUE (rval)) == type)
	    {
	      /* This is broken: it does not compute real
	         visibility information.  */
	      enum visibility_type new_v
		= compute_visibility (basetypes, TREE_VALUE (rval));
	      if (this_v != new_v)
		errstr = "conflicting visibilities to member `%s'";
	    }
	  /* Same baseclass, different places in the lattice.  */
	  else if (CLASSTYPE_MAIN_VARIANT (DECL_CONTEXT (TREE_VALUE (rval)))
		   == CLASSTYPE_MAIN_VARIANT (type))
	    errstr = "member `%s' belongs to distinct base classes `%s'";
	  else
	    {
	      tree nval = lookup_field_1 (type, name);

	      if (nval && get_base_type (type, DECL_CONTEXT (TREE_VALUE (rval))) == 0)
		{
		  /* We found it in other than a baseclass of RVAL's.  */
		  errstr = "request for member `%s' is ambiguous";
		}
	    }
	  if (errstr && entry)
	    {
	      tree error_string = my_build_string (errstr);
	      TREE_TYPE (entry) = error_string;
	    }
	  if (errstr && protect)
	    {
	      break;
	    }
	}
      else
	{
	  rval = lookup_fnfields_1 (type, name);
	  if (rval && (protect || type_stack))
	    {
	      /* @@ This is broken: it does not compute correct
		 visibility information.  */
	      this_v = compute_visibility (basetypes, TREE_VALUE (rval));
	      TREE_VALUE (entry) = rval;
	    }
	}
    }
  {
    tree *tp = search_stack->first;
    tree *search_tail = tp + tail;

#if 0
    /* If this FIELD_DECL defines its own visibility, deal with that.

       Arguments to compute_visibility are wrong!  */
    if (rval && errstr == 0
	&& DECL_VISIBILITY (TREE_VALUE (rval))
	&& (protect || type_stack))
      {
	assert (0);
	while (tp < search_tail)
	  {
	    /* If is possible for one of the derived types on the
	       path to have defined special visibility for this
	       field.  Look for such declarations and report an
	       error if a conflict is found.  */
	    enum visibility_type new_v;

	    if (this_v != visibility_default)
	      new_v = compute_visibility (tp[1], tp[0], TREE_VALUE (rval));
	    if (this_v != visibility_default && new_v != this_v)
	      {
		errstr = "conflicting visibilities to member `%s'";
		this_v = visibility_default;
	      }
	    CLASSTYPE_MARKED2 (*tp++) = 0;
	    tp++;
	  }
      }
    else
#endif
      {
	while (tp < search_tail)
	  {
	    CLASSTYPE_MARKED2 (*tp++) = 0;
	    tp++;
	  }
      }
  }
  search_stack = pop_search_level (search_stack, &search_obstack);

  if (errstr == 0)
    {
      if (this_v == visibility_private)
	errstr = TREE_PRIVATE (TREE_VALUE (rval))
	  ? "method `%s' is private" : "method `%s' is from private base class";
      else if (this_v == visibility_protected)
	errstr = "method `%s' is protected";
    }

  if (entry)
    {
      if (errstr)
	{
	  tree error_string = my_build_string (errstr);
	  /* Save error message with entry.  */
	  TREE_TYPE (entry) = error_string;
	}
      else
	{
	  /* Mark entry as having no error string.  */
	  TREE_TYPE (entry) = NULL_TREE;
	}
    }

  if (errstr && protect)
    {
      error (errstr, IDENTIFIER_POINTER (name), TYPE_NAME_STRING (type));
      rval = error_mark_node;
    }

  return rval;
}

/* BREADTH-FIRST SEARCH ROUTINES.  */
/* Search a multiple inheritance hierarchy by breadth-first search.

   TYPE is an aggregate type, possibly in a multiple-inheritance hierarchy.
   TESTFN is a function, which, if true, means that our condition has been met,
   and its return value should be returned.
   QFN, if non-NULL, is a predicate dictating whether the type should
   even be queued.  */

int
breadth_first_search (type, testfn, qfn)
     tree type;
     int (*testfn)();
     int (*qfn)();
{
  int head = 0, tail = 0;
  int rval = 0;

  search_stack = push_search_level (search_stack, &search_obstack);

  while (1)
    {
      int n_baselinks = CLASSTYPE_N_BASECLASSES (type);
      int i;

      /* Process and/or queue base types.  */
      for (i = 1; i <= n_baselinks; i++)
	if (CLASSTYPE_MARKED (CLASSTYPE_BASECLASS (type, i)) == 0
	    && (qfn == 0 || (*qfn) (CLASSTYPE_BASECLASS (type, i))))
	  {
	    CLASSTYPE_MARKED (CLASSTYPE_BASECLASS (type, i)) = 1;
	    obstack_grow (&search_obstack, &CLASSTYPE_BASECLASS (type, i), sizeof (tree*));
	    tail++;
	  }

      /* Process head of queue, if one exists.  */
      if (head >= tail)
	{
	  rval = 0;
	  break;
	}

      type = search_stack->first[head++];
      if (rval = (*testfn) (type))
	break;
    }
  {
    tree *tp = search_stack->first;
    tree *search_tail = tp + tail;
    while (tp < search_tail)
      {
	CLASSTYPE_MARKED (*tp++) = 0;
      }
  }

  search_stack = pop_search_level (search_stack, &search_obstack);
  return rval;
}

/* Functions to use in breadth first searches.  */
typedef tree (*pft)();
typedef int (*pfi)();

int tree_needs_constructor_p (type)
     tree type;
{
  return TREE_NEEDS_CONSTRUCTOR (type);
}

static tree declarator;

static tree get_virtuals_named_this (type)
     tree type;
{
  tree fields = lookup_fnfields (type, declarator);
  if (fields == 0 || fields == error_mark_node)
    return 0;
  if (TREE_VIRTUAL (TREE_VALUE (fields)))
    return fields;
  return 0;
}

pft get_virtual_named (decl)
     tree decl;
{
  declarator = decl;
  return get_virtuals_named_this;
}

tree get_virtual_destructor (type)
     tree type;
{
  if (TREE_HAS_DESTRUCTOR (type)
      && TREE_VIRTUAL (TREE_VALUE (TYPE_FN_FIELDS (type))))
    return TREE_VALUE (TYPE_FN_FIELDS (type));
  return 0;
}

int tree_has_any_destructor_p (type)
     tree type;
{
  return TREE_NEEDS_DESTRUCTOR (type);
}

/* Given a class type TYPE, and a function decl FNDECL,
   look for the first function the TYPE's heirarchy which
   FNDECL could match as a virtual function.

   DTORP is nonzero if we are looking for a destructor.  Destructors
   need special treatment because they do not match by name.  */
tree
get_first_matching_virtual (type, fndecl, dtorp)
     tree type, fndecl;
     int dtorp;
{
  tree tmp = NULL_TREE;

  /* Breadth first search routines start searching basetypes
     of TYPE, so we must perform first ply of search here.  */
  if (dtorp)
    {
      if (tree_has_any_destructor_p (type))
	tmp = get_virtual_destructor (type);

      if (tmp)
	return tmp;

      tmp = (tree) breadth_first_search (type,
					 get_virtual_destructor,
					 tree_has_any_destructor_p);
      return tmp;
    }
  else
    {
      tree tmp1, dtypes, btypes;
      int matched;

      declarator = DECL_ORIGINAL_NAME (fndecl);
      tmp = get_virtuals_named_this (type);

      if (tmp == 0)
	tmp = (tree) breadth_first_search (type, get_virtuals_named_this, 0);

      if (tmp == 0)
	return 0;

      tmp1 = TREE_VALUE (tmp);
      dtypes = TYPE_ARG_TYPES (TREE_TYPE (fndecl));
      while (tmp1)
	{
	  btypes = TYPE_ARG_TYPES (TREE_TYPE (TREE_TYPE (tmp1)));
	  if (compparms (TREE_CHAIN (btypes), TREE_CHAIN (dtypes), 1))
	    break;
	  else
	    {
	      tree b = TREE_CHAIN (btypes);
	      tree d = TREE_CHAIN (dtypes);
	      while (b && d)
		{
		  if (! comp_target_types (TREE_VALUE (b), TREE_VALUE (d), 1))
		    break;
		  b = TREE_CHAIN (b);
		  d = TREE_CHAIN (d);
		}
	      if (b == 0 && d == 0)
		{
		  error_with_decl (TREE_TYPE (tmp1), "conficting specification deriving from virtual function `%s'");
		  break;
		}
	    }
	  tmp1 = TREE_CHAIN (tmp1);
	}
      return tmp1;
    }
}

int
get_n_fnfield_sources (type, name)
{
  return 1;
}

int
get_n_field_sources (type, name)
{
  return 1;
}

/* DEPTH-FIRST SEARCH ROUTINES.  */

/* Assign unique numbers to _CLASSTYPE members of the lattice
   specified by TYPE.  The root nodes are marked first; the nodes
   are marked depth-fisrt, left-right.  */

static int cid;

/* Matrix implementing a relation from CLASSTYPE X CLASSTYPE => INT.
   Relation yields 1 if C1 <= C2, 0 otherwise.  */
typedef char mi_boolean;
static mi_boolean *mi_matrix;

/* Type for which this matrix is defined.  */
static tree mi_type;

/* Size of the matrix for indexing purposes.  */
static int mi_size;

/* Return nonzero if class C2 derives from class C1.  */
#define DERIVES_FROM(C1, C2)	\
  ((mi_matrix+mi_size*(CLASSTYPE_CID (C1)-1))[CLASSTYPE_CID (C2)-1])
#define DERIVES_FROM_STAR(C)	\
  (mi_matrix+(CLASSTYPE_CID (C)-1))

/* The main function which implements depth first search.  */
static void
dfs_walk (type, fn, qfn)
     tree type;
     void (*fn)();
     int (*qfn)();
{
  int i, n_baselinks = CLASSTYPE_N_BASECLASSES (type);

  for (i = 1; i <= n_baselinks; i++)
    if ((*qfn)(CLASSTYPE_BASECLASS (type, i)))
      {
	dfs_walk (CLASSTYPE_BASECLASS (type, i), fn, qfn);
      }

  fn (type);
}

/* Predicate functions which serve for dfs_walk.  */
static int numberedp (type) tree type;
{ return CLASSTYPE_CID (type); }
static int unnumberedp (type) tree type;
{ return CLASSTYPE_CID (type) == 0; }

static int markedp (type) tree type;
{ return CLASSTYPE_MARKED (type); }
static int unmarkedp (type) tree type;
{ return CLASSTYPE_MARKED (type) == 0; }
static int marked2p (type) tree type;
{ return CLASSTYPE_MARKED2 (type); }
static int unmarked2p (type) tree type;
{ return CLASSTYPE_MARKED2 (type) == 0; }
static int marked3p (type) tree type;
{ return CLASSTYPE_MARKED3 (type); }
static int unmarked3p (type) tree type;
{ return CLASSTYPE_MARKED3 (type) == 0; }
static int marked4p (type) tree type;
{ return CLASSTYPE_MARKED4 (type); }
static int unmarked4p (type) tree type;
{ return CLASSTYPE_MARKED4 (type) == 0; }

/* The worker functions for `dfs_walk'.  These do not need to
   test anything (vis a vis marking) if they are paired with
   a predicate function (above).  */

/* Assign each type within the lattice a number which is unique
   in the lattice.  The first number assigned is 1.  */

static void
dfs_number (type)
     tree type;
{
  CLASSTYPE_CID (type) = ++cid;
}

static void
dfs_unnumber (type)
     tree type;
{
  CLASSTYPE_CID (type) = 0;
}

static void
dfs_mark (type) tree type;
{ CLASSTYPE_MARKED (type) = 1; }

static void
dfs_unmark (type) tree type;
{ CLASSTYPE_MARKED (type) = 0; }

static void
dfs_mark2 (type) tree type;
{ CLASSTYPE_MARKED2 (type) = 1; }

static void
dfs_unmark2 (type) tree type;
{ CLASSTYPE_MARKED2 (type) = 0; }

static void
dfs_mark3 (type) tree type;
{ CLASSTYPE_MARKED3 (type) = 1; }

static void
dfs_unmark3 (type) tree type;
{ CLASSTYPE_MARKED3 (type) = 0; }

static void
dfs_mark4 (type) tree type;
{ CLASSTYPE_MARKED4 (type) = 1; }

static void
dfs_unmark4 (type) tree type;
{ CLASSTYPE_MARKED4 (type) = 0; }

static void
dfs_record_inheritance (type)
     tree type;
{
  int i, n_baselinks = CLASSTYPE_N_BASECLASSES (type);
  mi_boolean *derived_row = DERIVES_FROM_STAR (type);

  for (i = n_baselinks; i > 0; i--)
    {
      int j;
      tree baseclass = CLASSTYPE_BASECLASS (type, i);
      mi_boolean *base_row = DERIVES_FROM_STAR (baseclass);

      /* Don't search if there's nothing there!  */
      if (CLASSTYPE_N_BASECLASSES (baseclass))
	for (j = mi_size*(CLASSTYPE_CID (baseclass)-1); j >= 0; j -= mi_size)
	  derived_row[j] |= base_row[j];
      DERIVES_FROM (baseclass, type) = 1;
    }

  CLASSTYPE_MARKED (type) = 1;
}

/* Given a _CLASSTYPE node in a multiple inheritance lattice,
   convert the lattice into a simple relation such that,
   given to CIDs, C1 and C2, one can determine if C1 <= C2
   or C2 <= C1 or C1 <> C2.

   Once constructed, we walk the lattice depth fisrt,
   applying various functions to elements as they are encountered.  */

void
build_mi_matrix (type)
     tree type;
{
  cid = 0;
  mi_size = CLASSTYPE_N_SUPERCLASSES (type);
  mi_matrix = (char *)malloc (mi_size * mi_size);
  mi_type = type;
  bzero (mi_matrix, mi_size * mi_size);
  dfs_walk (type, dfs_number, unnumberedp);
  dfs_walk (type, dfs_record_inheritance, unmarkedp);
  dfs_walk (type, dfs_unmark, markedp);
}

void
free_mi_matrix ()
{
  dfs_walk (mi_type, dfs_unnumber, numberedp);
  free (mi_matrix);
  mi_size = 0;
  cid = 0;
}

/* Subroutines of push_class_decls ().  */

/* Add the instance variables which this class contributed to the
   current class binding contour.  When a redefinition occurs,
   if the redefinition is strictly within a single inheritance path,
   we just overwrite (in the case of a data field) or
   cons (in the case of a member function) the old declaration with
   the new.  If the fields are not within a single inheritance path,
   we must cons them in either case.  */

static void
dfs_pushdecls (type)
     tree type;
{
  tree fields;

  for (fields = TYPE_FIELDS (type); fields; fields = TREE_CHAIN (fields))
    {
      /* Unmark so that if we are in a constructor, and then find that
	 this field was initialized by a base initializer,
	 we can emit an error message.  */
      TREE_EVERUSED (fields) = 0;
      if (DECL_NAME (fields))
	{
	  tree value = IDENTIFIER_CLASS_VALUE (DECL_NAME (fields));
	  if (value)
	    {
	      /* Possible ambiguity.  If its defining type(s)
		 is (are all) derived from us, no problem.  */

	      if (TREE_CODE (value) == FIELD_DECL)
		value = DERIVES_FROM (DECL_CONTEXT (value), type)
		  ? fields : tree_cons (NULL_TREE, fields,
					build_tree_list (NULL_TREE, value));
	      else
		{
		  /* All children may derive from us.  */
		  int i, n_children = list_length (value);

		  /* This code is clearly wrong, but I can't remember
		     what is clearly right.  */
		  assert (0);
		  for (i = 0; i < n_children; i++)
		    if (! DERIVES_FROM (DECL_CONTEXT (TREE_VALUE (value)), type))
		      break;

		  value = (i == n_children) ? fields : tree_cons (NULL_TREE, fields, value);
		}

	      /* Mark this as a potentially ambiguous member.  */
	      if (TREE_CODE (value) == TREE_LIST)
		{
		  /* Leaving TREE_TYPE blank is intentional.
		     We cannot use `error_mark_node' (lookup_name)
		     or `unknown_type_node' (all member functions use this).  */
		  TREE_NONLOCAL (value) = 1;
		}

	      IDENTIFIER_CLASS_VALUE (DECL_NAME (fields)) = value;
	    }
	  else IDENTIFIER_CLASS_VALUE (DECL_NAME (fields)) = fields;
	}
    }

  fields = TYPE_FN_FIELDS (type);

  /* Farm out constructors and destructors.  */

  if (TREE_HAS_CONSTRUCTOR (type))
    fields = TREE_CHAIN (fields);

  /* This does not work for multiple inheritance yet.  */
  while (fields)
    {
      /* This will cause lookup_name to return a pointer
	 to the tree_list of possible methods of this name.
	 If the order is a problem, we can nreverse them.  */
      tree tmp;
      tree old = IDENTIFIER_CLASS_VALUE (TREE_PURPOSE (fields));

      if (old && TREE_CODE (old) == TREE_LIST)
	tmp = tree_cons (TREE_PURPOSE (fields), fields, old);
      else
	{
	  if (old)
	    warning ("shadowing member `%s' with member function",
		     IDENTIFIER_POINTER (TREE_PURPOSE (fields)));
	  tmp = build_tree_list (TREE_PURPOSE (fields), fields);
	}

      TREE_TYPE (tmp) = unknown_type_node;
      TREE_OVERLOADED (tmp) = TREE_OVERLOADED (fields);
      IDENTIFIER_CLASS_VALUE (TREE_PURPOSE (fields)) = tmp;
      fields = TREE_CHAIN (fields);
    }

  CLASSTYPE_MARKED (type) = 1;
}

/* Consolidate unique (by name) member functions.  */
static void
dfs_compress_decls (type)
     tree type;
{
  tree fields = TYPE_FN_FIELDS (type);

  /* Farm out constructors and destructors.  */
  if (TREE_HAS_CONSTRUCTOR (type))
    fields = TREE_CHAIN (fields);

  while (fields)
    {
      tree tmp = IDENTIFIER_CLASS_VALUE (TREE_PURPOSE (fields));
      if (TREE_CHAIN (tmp) == NULL_TREE
	  && TREE_VALUE (TREE_VALUE (tmp))
	  && TREE_CHAIN (TREE_VALUE (TREE_VALUE (tmp))) == NULL_TREE)
	{
	  IDENTIFIER_CLASS_VALUE (TREE_PURPOSE (fields))
	    = TREE_TYPE (TREE_VALUE (TREE_VALUE (tmp)));
	}
      fields = TREE_CHAIN (fields);
    }

  CLASSTYPE_MARKED (type) = 0;
}

/* When entering the scope of a class, we cache all of the
   fields that that class provides within its inheritance
   lattice.  Where ambiguities result, we mark them
   with `error_mark_node' so that if they are encountered
   without explicit qualification, we can emit an error
   message.  */
void
push_class_decls (type)
     tree type;
{
#if 0
  tree tags = TYPE_TAGS (type);

  while (tags)
    {
      tree code_type_node;
      tree tag;

      switch (TREE_CODE (TREE_VALUE (tags)))
	{
	case ENUMERAL_TYPE:
	  code_type_node = enum_type_node;
	  break;
	case RECORD_TYPE:
	  code_type_node = record_type_node;
	  break;
	case CLASS_TYPE:
	  code_type_node = class_type_node;
	  break;
	case UNION_TYPE:
	  code_type_node = union_type_node;
	  break;
	default:
	  assert (0);
	}
      tag = xref_tag (code_type_node, TREE_PURPOSE (tags),
		      CLASSTYPE_BASECLASS (TREE_VALUE (tags), 1));
      pushdecl (build_decl (TYPE_DECL, TREE_PURPOSE (tags), TREE_VALUE (tags)));
    }
#endif

  /* Push class fields into CLASS_VALUE scope, and mark.  */
  dfs_walk (type, dfs_pushdecls, unmarkedp);

  /* Compress fields which have only a single entry
     by a given name, and unmark.  */
  dfs_walk (type, dfs_compress_decls, markedp);
}

static void
dfs_popdecls (type)
     tree type;
{
  tree fields = TYPE_FIELDS (type);

  while (fields)
    {
      if (DECL_NAME (fields))
	IDENTIFIER_CLASS_VALUE (DECL_NAME (fields)) = NULL_TREE;
      fields = TREE_CHAIN (fields);
    }
  for (fields = TYPE_FN_FIELDS (type); fields; fields = TREE_CHAIN (fields))
    {
      IDENTIFIER_CLASS_VALUE (TREE_PURPOSE (fields)) = NULL_TREE;
    }

  CLASSTYPE_MARKED (type) = 1;
}

void
pop_class_decls (type)
     tree type;
{
  /* Clear out the IDENTIFIER_CLASS_VALUE which this
     class may have occupied, and mark.  */
  dfs_walk (type, dfs_popdecls, unmarkedp);

  /* Unmark.  */
  dfs_walk (type, dfs_unmark, markedp);
}

/* Given a base type PARENT, and a derived type TYPE, build
   a name which distinguishes exactly the PARENT member of TYPE's type.

   FORMAT is a string which controls how sprintf formats the name
   we have generated.

   For example, given

	class A; class B; class C : A, B;

   it is possible to distinguish "A" from "C's A".  And given

	class L;
	class A : L; class B : L; class C : A, B;

   it is possible to distinguish "L" from "A's L", and also from
   "C's L from A".  */
tree
build_type_pathname (format, parent, type)
     char *format;
     tree parent, type;
{
  char *base = obstack_base (&search_obstack);
  char *name;
  int i;
  tree id;

  do
    {
      obstack_grow (&search_obstack,
		    TYPE_NAME_STRING (type), TYPE_NAME_LENGTH (type));
      obstack_1grow (&search_obstack, '_');
      for (i = CLASSTYPE_N_BASECLASSES (type); i > 0; i++)
	if (! tree_int_cst_lt (CLASSTYPE_OFFSET (parent),
			       CLASSTYPE_OFFSET (CLASSTYPE_BASECLASS (type, i))))
	  {
	    type = CLASSTYPE_BASECLASS (type, i);
	    break;
	  }
      if (i == 0) abort ();
    } while (type != parent);
  obstack_grow0 (&search_obstack,
		 TYPE_NAME_STRING (type), TYPE_NAME_LENGTH (type));
  i = obstack_object_size (&search_obstack);
  name = obstack_base (&search_obstack) + i;
  obstack_blank (&search_obstack, strlen (format) + i + 1);
  sprintf (name, format, base);
  id = get_identifier (name);
  obstack_free (&search_obstack, base);
  return id;
}

void
unmark_finished_struct (type)
{
  dfs_walk (type, dfs_unmark4, marked4p);
}

#include <stdio.h>

void
print_search_statistics ()
{
  if (flag_memoize_lookups)
    {
      fprintf (stderr, "%d memoized contexts saved\n",
	       n_contexts_saved);
      fprintf (stderr, "%d local tree nodes made\n", my_tree_node_counter);
      fprintf (stderr, "%d local hash nodes made\n", my_memoized_entry_counter);
      fprintf (stderr, "fields statistics:\n");
      fprintf (stderr, "  memoized finds = %d; rejects = %d; (searches = %d)\n",
	       memoized_fast_finds[0], memoized_fast_rejects[0],
	       memoized_fields_searched[0]);
      fprintf (stderr, "  memoized_adds = %d\n", memoized_adds[0]);
      fprintf (stderr, "fnfields statistics:\n");
      fprintf (stderr, "  memoized finds = %d; rejects = %d; (searches = %d)\n",
	       memoized_fast_finds[1], memoized_fast_rejects[1],
	       memoized_fields_searched[1]);
      fprintf (stderr, "  memoized_adds = %d\n", memoized_adds[1]);
    }
  fprintf (stderr, "%d fields searched in %d[%d] calls to lookup_field[_1]\n",
	   n_fields_searched, n_calls_lookup_field, n_calls_lookup_field_1);
  fprintf (stderr, "%d fnfields searched in %d calls to lookup_fnfields\n",
	   n_outer_fields_searched, n_calls_lookup_fnfields);
}

void init_search ()
{
  obstack_init (&search_obstack);
  obstack_init (&type_obstack);
  obstack_init (&type_obstack_entries);
}
