#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <inttypes.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>

#include "luks.h"
#include "../lib/libcryptsetup.h"
#include "../lib/internal.h"
#include "../lib/blockdev.h"

static int setup_mapping(const char *cipher, const char *name, 
			 const char *device, unsigned int payloadOffset,
			 const char *key, unsigned int keyLength, 
			 unsigned int sector, unsigned int srcLength, 
			 struct setup_backend *backend)
{
	struct crypt_options k;
	struct crypt_options *options = &k;

	int r;

	options->offset = sector;
	options->size = payloadOffset;	

	options->cipher = cipher;
	options->key_size = keyLength;
	options->skip = 0; options->flags = 0;
	options->name = name;
	options->device = device;
	
	if (options->size <= options->offset) {
		set_error("Invalid offset");
		return -EINVAL;
	}

	r = backend->create(0, options, key);

	if (r <= 0)
		set_error(NULL);

	return r;
}

static int clear_mapping(const char *name, struct setup_backend *backend)
{
	struct crypt_options options;
	options.name=name;
	backend->remove(&options);	
}

static int LUKS_endec_template(char *src, unsigned int srcLength, 
			       struct luks_phdr *hdr, 
			       char *key, unsigned int keyLength, 
			       const char *device, 
			       unsigned int sector, struct setup_backend *backend,
			       ssize_t (*func)(int, void *, size_t))
{
	int devfd;
	char *name;
	char *fullpath;
	char *dmCipherSpec;
	int r;
	
	asprintf(&name,"temporary-cryptsetup-%d",getpid());
	asprintf(&fullpath,"%s/%s",dm_dir(),name);
	asprintf(&dmCipherSpec,"%s-%s",hdr->cipherName, hdr->cipherMode);

	r = setup_mapping(dmCipherSpec,name,device,hdr->payloadOffset,key,keyLength,sector,srcLength,backend);
	if(r < 0) {
		fprintf(stderr,"failed to setup dm-crypt mapping.\n");
		goto out; 
	}

	devfd = open(fullpath,O_RDWR | O_DIRECT | O_SYNC);
	if(devfd == -1) { r = -EIO; goto out2; }

	r = func(devfd,src,srcLength);
	if(r < 0) { r = -EIO; goto out3; }

	r = 0;
out3:
	close(devfd);
out2:
	clear_mapping(name,backend);
out:
	free(name); free(fullpath); free(dmCipherSpec);
	return r;
}

int LUKS_encrypt_to_storage(char *src, unsigned int srcLength, 
			    struct luks_phdr *hdr, 
			    char *key, unsigned int keyLength, 
			    const char *device, 
			    unsigned int sector, struct setup_backend *backend)
{
	
	return LUKS_endec_template(src,srcLength,hdr,key,keyLength, device, sector, backend,	
				   (ssize_t (*)(int, void *, size_t)) write_blockwise);
}	

int LUKS_decrypt_from_storage(char *dst, unsigned int dstLength, 
			      struct luks_phdr *hdr, 
			      char *key, unsigned int keyLength, 
			      const char *device, 
			      unsigned int sector, struct setup_backend *backend)
{
	return LUKS_endec_template(dst,dstLength,hdr,key,keyLength, device, sector, backend, read_blockwise);
}
