/* SPDX-License-Identifier: MIT */
/*
 * Copyright © 2023 Intel Corporation
 */

#ifndef _XE_GT_REGS_H_
#define _XE_GT_REGS_H_

#include "regs/xe_reg_defs.h"

/*
 * The GSI register range [0x0 - 0x40000) is replicated at a higher offset
 * for the media GT.  xe_mmio and xe_gt_mcr functions will automatically
 * translate offsets by MEDIA_GT_GSI_OFFSET when operating on the media GT.
 */
#define MEDIA_GT_GSI_OFFSET				0x380000
#define MEDIA_GT_GSI_LENGTH				0x40000

/* MTL workpoint reg to get core C state and actual freq of 3D, SAMedia */
#define MTL_MIRROR_TARGET_WP1				XE_REG(0xc60)
#define   MTL_CAGF_MASK					REG_GENMASK(8, 0)
#define   MTL_CC_MASK					REG_GENMASK(12, 9)

/* RPM unit config (Gen8+) */
#define RPM_CONFIG0					XE_REG(0xd00)
#define   RPM_CONFIG0_CRYSTAL_CLOCK_FREQ_MASK		REG_GENMASK(5, 3)
#define   RPM_CONFIG0_CRYSTAL_CLOCK_FREQ_24_MHZ		0
#define   RPM_CONFIG0_CRYSTAL_CLOCK_FREQ_19_2_MHZ	1
#define   RPM_CONFIG0_CRYSTAL_CLOCK_FREQ_38_4_MHZ	2
#define   RPM_CONFIG0_CRYSTAL_CLOCK_FREQ_25_MHZ		3
#define   RPM_CONFIG0_CTC_SHIFT_PARAMETER_MASK		REG_GENMASK(2, 1)

#define FORCEWAKE_ACK_MEDIA_VDBOX(n)		XE_REG(0xd50 + (n) * 4)
#define FORCEWAKE_ACK_MEDIA_VEBOX(n)		XE_REG(0xd70 + (n) * 4)
#define FORCEWAKE_ACK_RENDER			XE_REG(0xd84)

#define GMD_ID					XE_REG(0xd8c)
#define   GMD_ID_ARCH_MASK			REG_GENMASK(31, 22)
#define   GMD_ID_RELEASE_MASK			REG_GENMASK(21, 14)
#define   GMD_ID_REVID				REG_GENMASK(5, 0)

#define FORCEWAKE_ACK_GSC			XE_REG(0xdf8)
#define FORCEWAKE_ACK_GT_MTL			XE_REG(0xdfc)

#define MCFG_MCR_SELECTOR			XE_REG(0xfd0)
#define MTL_MCR_SELECTOR			XE_REG(0xfd4)
#define SF_MCR_SELECTOR				XE_REG(0xfd8)
#define MCR_SELECTOR				XE_REG(0xfdc)
#define GAM_MCR_SELECTOR			XE_REG(0xfe0)
#define   MCR_MULTICAST				REG_BIT(31)
#define   MCR_SLICE_MASK			REG_GENMASK(30, 27)
#define   MCR_SLICE(slice)			REG_FIELD_PREP(MCR_SLICE_MASK, slice)
#define   MCR_SUBSLICE_MASK			REG_GENMASK(26, 24)
#define   MCR_SUBSLICE(subslice)		REG_FIELD_PREP(MCR_SUBSLICE_MASK, subslice)
#define   MTL_MCR_GROUPID			REG_GENMASK(11, 8)
#define   MTL_MCR_INSTANCEID			REG_GENMASK(3, 0)

#define PS_INVOCATION_COUNT			XE_REG(0x2348)

#define XELP_GLOBAL_MOCS(i)			XE_REG(0x4000 + (i) * 4)
#define XEHP_GLOBAL_MOCS(i)			XE_REG_MCR(0x4000 + (i) * 4)
#define   LE_SSE_MASK				REG_GENMASK(18, 17)
#define   LE_SSE(value)				REG_FIELD_PREP(LE_SSE_MASK, value)
#define   LE_COS_MASK				REG_GENMASK(16, 15)
#define   LE_COS(value)				REG_FIELD_PREP(LE_COS_MASK)
#define   LE_SCF_MASK				REG_BIT(14)
#define   LE_SCF(value)				REG_FIELD_PREP(LE_SCF_MASK, value)
#define   LE_PFM_MASK				REG_GENMASK(13, 11)
#define   LE_PFM(value)				REG_FIELD_PREP(LE_PFM_MASK, value)
#define   LE_SCC_MASK				REG_GENMASK(10, 8)
#define   LE_SCC(value)				REG_FIELD_PREP(LE_SCC_MASK, value)
#define   LE_RSC_MASK				REG_BIT(7)
#define   LE_RSC(value)				REG_FIELD_PREP(LE_RSC_MASK, value)
#define   LE_AOM_MASK				REG_BIT(6)
#define   LE_AOM(value)				REG_FIELD_PREP(LE_AOM_MASK, value)
#define   LE_LRUM_MASK				REG_GENMASK(5, 4)
#define   LE_LRUM(value)			REG_FIELD_PREP(LE_LRUM_MASK, value)
#define   LE_TGT_CACHE_MASK			REG_GENMASK(3, 2)
#define   LE_TGT_CACHE(value)			REG_FIELD_PREP(LE_TGT_CACHE_MASK, value)
#define   LE_CACHEABILITY_MASK			REG_GENMASK(1, 0)
#define   LE_CACHEABILITY(value)		REG_FIELD_PREP(LE_CACHEABILITY_MASK, value)

#define STATELESS_COMPRESSION_CTRL		XE_REG_MCR(0x4148)
#define   UNIFIED_COMPRESSION_FORMAT		REG_GENMASK(3, 0)

#define XE2_GAMREQSTRM_CTRL			XE_REG_MCR(0x4194)
#define   CG_DIS_CNTLBUS			REG_BIT(6)

#define CCS_AUX_INV				XE_REG(0x4208)

#define VD0_AUX_INV				XE_REG(0x4218)
#define VE0_AUX_INV				XE_REG(0x4238)

#define VE1_AUX_INV				XE_REG(0x42b8)
#define   AUX_INV				REG_BIT(0)

#define XE2_LMEM_CFG				XE_REG(0x48b0)

#define XEHP_TILE_ADDR_RANGE(_idx)		XE_REG_MCR(0x4900 + (_idx) * 4)
#define XEHP_FLAT_CCS_BASE_ADDR			XE_REG_MCR(0x4910)
#define XEHP_FLAT_CCS_PTR			REG_GENMASK(31, 8)

#define WM_CHICKEN3				XE_REG_MCR(0x5588, XE_REG_OPTION_MASKED)
#define   HIZ_PLANE_COMPRESSION_DIS		REG_BIT(10)

#define CHICKEN_RASTER_1			XE_REG_MCR(0x6204, XE_REG_OPTION_MASKED)
#define   DIS_SF_ROUND_NEAREST_EVEN		REG_BIT(8)
#define   DIS_CLIP_NEGATIVE_BOUNDING_BOX	REG_BIT(6)

#define CHICKEN_RASTER_2			XE_REG_MCR(0x6208, XE_REG_OPTION_MASKED)
#define   TBIMR_FAST_CLIP			REG_BIT(5)

#define FF_MODE					XE_REG_MCR(0x6210)
#define   DIS_TE_AUTOSTRIP			REG_BIT(31)
#define   VS_HIT_MAX_VALUE_MASK			REG_GENMASK(25, 20)
#define   DIS_MESH_PARTIAL_AUTOSTRIP		REG_BIT(16)
#define   DIS_MESH_AUTOSTRIP			REG_BIT(15)

#define VFLSKPD					XE_REG_MCR(0x62a8, XE_REG_OPTION_MASKED)
#define   DIS_PARTIAL_AUTOSTRIP			REG_BIT(9)
#define   DIS_AUTOSTRIP				REG_BIT(6)
#define   DIS_OVER_FETCH_CACHE			REG_BIT(1)
#define   DIS_MULT_MISS_RD_SQUASH		REG_BIT(0)

#define FF_MODE2				XE_REG(0x6604)
#define XEHP_FF_MODE2				XE_REG_MCR(0x6604)
#define   FF_MODE2_GS_TIMER_MASK		REG_GENMASK(31, 24)
#define   FF_MODE2_GS_TIMER_224			REG_FIELD_PREP(FF_MODE2_GS_TIMER_MASK, 224)
#define   FF_MODE2_TDS_TIMER_MASK		REG_GENMASK(23, 16)
#define   FF_MODE2_TDS_TIMER_128		REG_FIELD_PREP(FF_MODE2_TDS_TIMER_MASK, 4)

#define XEHPG_INSTDONE_GEOM_SVGUNIT		XE_REG_MCR(0x666c)

#define CACHE_MODE_1				XE_REG(0x7004, XE_REG_OPTION_MASKED)
#define   MSAA_OPTIMIZATION_REDUC_DISABLE	REG_BIT(11)

#define COMMON_SLICE_CHICKEN1			XE_REG(0x7010, XE_REG_OPTION_MASKED)
#define   DISABLE_BOTTOM_CLIP_RECTANGLE_TEST	REG_BIT(14)

#define HIZ_CHICKEN					XE_REG(0x7018, XE_REG_OPTION_MASKED)
#define   DG1_HZ_READ_SUPPRESSION_OPTIMIZATION_DISABLE	REG_BIT(14)
#define   HZ_DEPTH_TEST_LE_GE_OPT_DISABLE		REG_BIT(13)

#define XEHP_PSS_MODE2				XE_REG_MCR(0x703c, XE_REG_OPTION_MASKED)
#define   SCOREBOARD_STALL_FLUSH_CONTROL	REG_BIT(5)

#define XEHP_PSS_CHICKEN			XE_REG_MCR(0x7044, XE_REG_OPTION_MASKED)
#define   FLSH_IGNORES_PSD			REG_BIT(10)
#define   FD_END_COLLECT			REG_BIT(5)

#define SC_INSTDONE				XE_REG(0x7100)
#define SC_INSTDONE_EXTRA			XE_REG(0x7104)
#define SC_INSTDONE_EXTRA2			XE_REG(0x7108)

#define XEHPG_SC_INSTDONE			XE_REG_MCR(0x7100)
#define XEHPG_SC_INSTDONE_EXTRA			XE_REG_MCR(0x7104)
#define XEHPG_SC_INSTDONE_EXTRA2		XE_REG_MCR(0x7108)

#define COMMON_SLICE_CHICKEN4			XE_REG(0x7300, XE_REG_OPTION_MASKED)
#define   DISABLE_TDC_LOAD_BALANCING_CALC	REG_BIT(6)

#define COMMON_SLICE_CHICKEN3				XE_REG(0x7304, XE_REG_OPTION_MASKED)
#define XEHP_COMMON_SLICE_CHICKEN3			XE_REG_MCR(0x7304, XE_REG_OPTION_MASKED)
#define   DG1_FLOAT_POINT_BLEND_OPT_STRICT_MODE_EN	REG_BIT(12)
#define   XEHP_DUAL_SIMD8_SEQ_MERGE_DISABLE		REG_BIT(12)
#define   BLEND_EMB_FIX_DISABLE_IN_RCC			REG_BIT(11)
#define   DISABLE_CPS_AWARE_COLOR_PIPE			REG_BIT(9)

#define XEHP_SLICE_COMMON_ECO_CHICKEN1		XE_REG_MCR(0x731c, XE_REG_OPTION_MASKED)
#define   MSC_MSAA_REODER_BUF_BYPASS_DISABLE	REG_BIT(14)

#define VF_PREEMPTION				XE_REG(0x83a4, XE_REG_OPTION_MASKED)
#define   PREEMPTION_VERTEX_COUNT		REG_GENMASK(15, 0)

#define VF_SCRATCHPAD				XE_REG(0x83a8, XE_REG_OPTION_MASKED)
#define   XE2_VFG_TED_CREDIT_INTERFACE_DISABLE	REG_BIT(13)

#define VFG_PREEMPTION_CHICKEN			XE_REG(0x83b4, XE_REG_OPTION_MASKED)
#define   POLYGON_TRIFAN_LINELOOP_DISABLE	REG_BIT(4)

#define SQCNT1					XE_REG_MCR(0x8718)
#define XELPMP_SQCNT1				XE_REG(0x8718)
#define   SQCNT1_PMON_ENABLE			REG_BIT(30)
#define   SQCNT1_OABPC				REG_BIT(29)
#define   ENFORCE_RAR				REG_BIT(23)

#define XEHP_SQCM				XE_REG_MCR(0x8724)
#define   EN_32B_ACCESS				REG_BIT(30)

#define XE2_FLAT_CCS_BASE_RANGE_LOWER		XE_REG_MCR(0x8800)
#define   XE2_FLAT_CCS_ENABLE			REG_BIT(0)
#define XE2_FLAT_CCS_BASE_LOWER_ADDR_MASK	REG_GENMASK(31, 6)

#define XE2_FLAT_CCS_BASE_RANGE_UPPER		XE_REG_MCR(0x8804)
#define XE2_FLAT_CCS_BASE_UPPER_ADDR_MASK	REG_GENMASK(7, 0)

#define GSCPSMI_BASE				XE_REG(0x880c)

#define CCCHKNREG1				XE_REG_MCR(0x8828)
#define   L3CMPCTRL				REG_BIT(23)
#define   ENCOMPPERFFIX				REG_BIT(18)

/* Fuse readout registers for GT */
#define XEHP_FUSE4				XE_REG(0x9114)
#define   CFEG_WMTP_DISABLE			REG_BIT(20)
#define   CCS_EN_MASK				REG_GENMASK(19, 16)
#define   GT_L3_EXC_MASK			REG_GENMASK(6, 4)

#define	MIRROR_FUSE3				XE_REG(0x9118)
#define   XE2_NODE_ENABLE_MASK			REG_GENMASK(31, 16)
#define   L3BANK_PAIR_COUNT			4
#define   XEHPC_GT_L3_MODE_MASK			REG_GENMASK(7, 4)
#define   XE2_GT_L3_MODE_MASK			REG_GENMASK(7, 4)
#define   L3BANK_MASK				REG_GENMASK(3, 0)
#define   XELP_GT_L3_MODE_MASK			REG_GENMASK(7, 0)
/* on Xe_HP the same fuses indicates mslices instead of L3 banks */
#define   MAX_MSLICES				4
#define   MEML3_EN_MASK				REG_GENMASK(3, 0)

#define MIRROR_FUSE1				XE_REG(0x911c)

#define XELP_EU_ENABLE				XE_REG(0x9134)	/* "_DISABLE" on Xe_LP */
#define   XELP_EU_MASK				REG_GENMASK(7, 0)
#define XELP_GT_SLICE_ENABLE			XE_REG(0x9138)
#define XELP_GT_GEOMETRY_DSS_ENABLE		XE_REG(0x913c)

#define GT_VEBOX_VDBOX_DISABLE			XE_REG(0x9140)
#define   GT_VEBOX_DISABLE_MASK			REG_GENMASK(19, 16)
#define   GT_VDBOX_DISABLE_MASK			REG_GENMASK(7, 0)

#define XEHP_GT_COMPUTE_DSS_ENABLE		XE_REG(0x9144)
#define XEHPC_GT_COMPUTE_DSS_ENABLE_EXT		XE_REG(0x9148)
#define XE2_GT_COMPUTE_DSS_2			XE_REG(0x914c)
#define XE2_GT_GEOMETRY_DSS_1			XE_REG(0x9150)
#define XE2_GT_GEOMETRY_DSS_2			XE_REG(0x9154)

#define GDRST					XE_REG(0x941c)
#define   GRDOM_GUC				REG_BIT(3)
#define   GRDOM_FULL				REG_BIT(0)

#define MISCCPCTL				XE_REG(0x9424)
#define   DOP_CLOCK_GATE_RENDER_ENABLE		REG_BIT(1)

#define UNSLCGCTL9430				XE_REG(0x9430)
#define   MSQDUNIT_CLKGATE_DIS			REG_BIT(3)

#define UNSLICE_UNIT_LEVEL_CLKGATE		XE_REG(0x9434)
#define   VFUNIT_CLKGATE_DIS			REG_BIT(20)
#define   TSGUNIT_CLKGATE_DIS			REG_BIT(17) /* XEHPSDV */
#define   CG3DDISCFEG_CLKGATE_DIS		REG_BIT(17) /* DG2 */
#define   GAMEDIA_CLKGATE_DIS			REG_BIT(11)
#define   HSUNIT_CLKGATE_DIS			REG_BIT(8)
#define   VSUNIT_CLKGATE_DIS			REG_BIT(3)

#define UNSLCGCTL9440				XE_REG(0x9440)
#define   GAMTLBOACS_CLKGATE_DIS		REG_BIT(28)
#define   GAMTLBVDBOX5_CLKGATE_DIS		REG_BIT(27)
#define   GAMTLBVDBOX6_CLKGATE_DIS		REG_BIT(26)
#define   GAMTLBVDBOX3_CLKGATE_DIS		REG_BIT(24)
#define   GAMTLBVDBOX4_CLKGATE_DIS		REG_BIT(23)
#define   GAMTLBVDBOX7_CLKGATE_DIS		REG_BIT(22)
#define   GAMTLBVDBOX2_CLKGATE_DIS		REG_BIT(21)
#define   GAMTLBVDBOX0_CLKGATE_DIS		REG_BIT(17)
#define   GAMTLBKCR_CLKGATE_DIS			REG_BIT(16)
#define   GAMTLBGUC_CLKGATE_DIS			REG_BIT(15)
#define   GAMTLBBLT_CLKGATE_DIS			REG_BIT(14)
#define   GAMTLBVDBOX1_CLKGATE_DIS		REG_BIT(6)

#define UNSLCGCTL9444				XE_REG(0x9444)
#define   GAMTLBGFXA0_CLKGATE_DIS		REG_BIT(30)
#define   GAMTLBGFXA1_CLKGATE_DIS		REG_BIT(29)
#define   GAMTLBCOMPA0_CLKGATE_DIS		REG_BIT(28)
#define   GAMTLBCOMPA1_CLKGATE_DIS		REG_BIT(27)
#define   GAMTLBCOMPB0_CLKGATE_DIS		REG_BIT(26)
#define   GAMTLBCOMPB1_CLKGATE_DIS		REG_BIT(25)
#define   GAMTLBCOMPC0_CLKGATE_DIS		REG_BIT(24)
#define   GAMTLBCOMPC1_CLKGATE_DIS		REG_BIT(23)
#define   GAMTLBCOMPD0_CLKGATE_DIS		REG_BIT(22)
#define   GAMTLBCOMPD1_CLKGATE_DIS		REG_BIT(21)
#define   GAMTLBMERT_CLKGATE_DIS		REG_BIT(20)
#define   GAMTLBVEBOX3_CLKGATE_DIS		REG_BIT(19)
#define   GAMTLBVEBOX2_CLKGATE_DIS		REG_BIT(18)
#define   GAMTLBVEBOX1_CLKGATE_DIS		REG_BIT(17)
#define   GAMTLBVEBOX0_CLKGATE_DIS		REG_BIT(16)
#define   LTCDD_CLKGATE_DIS			REG_BIT(10)

#define XEHP_SLICE_UNIT_LEVEL_CLKGATE		XE_REG_MCR(0x94d4)
#define   L3_CR2X_CLKGATE_DIS			REG_BIT(17)
#define   L3_CLKGATE_DIS			REG_BIT(16)
#define   NODEDSS_CLKGATE_DIS			REG_BIT(12)
#define   MSCUNIT_CLKGATE_DIS			REG_BIT(10)
#define   RCCUNIT_CLKGATE_DIS			REG_BIT(7)
#define   SARBUNIT_CLKGATE_DIS			REG_BIT(5)
#define   SBEUNIT_CLKGATE_DIS			REG_BIT(4)

#define UNSLICE_UNIT_LEVEL_CLKGATE2		XE_REG(0x94e4)
#define   VSUNIT_CLKGATE2_DIS			REG_BIT(19)

#define SUBSLICE_UNIT_LEVEL_CLKGATE		XE_REG_MCR(0x9524)
#define   DSS_ROUTER_CLKGATE_DIS		REG_BIT(28)
#define   GWUNIT_CLKGATE_DIS			REG_BIT(16)

#define SUBSLICE_UNIT_LEVEL_CLKGATE2		XE_REG_MCR(0x9528)
#define   CPSSUNIT_CLKGATE_DIS			REG_BIT(9)

#define SSMCGCTL9530				XE_REG_MCR(0x9530)
#define   RTFUNIT_CLKGATE_DIS			REG_BIT(18)

#define DFR_RATIO_EN_AND_CHICKEN		XE_REG_MCR(0x9550)
#define   DFR_DISABLE				REG_BIT(9)

#define RPNSWREQ				XE_REG(0xa008)
#define   REQ_RATIO_MASK			REG_GENMASK(31, 23)

#define RP_CONTROL				XE_REG(0xa024)
#define   RPSWCTL_MASK				REG_GENMASK(10, 9)
#define   RPSWCTL_ENABLE			REG_FIELD_PREP(RPSWCTL_MASK, 2)
#define   RPSWCTL_DISABLE			REG_FIELD_PREP(RPSWCTL_MASK, 0)
#define RC_CONTROL				XE_REG(0xa090)
#define   RC_CTL_HW_ENABLE			REG_BIT(31)
#define   RC_CTL_TO_MODE			REG_BIT(28)
#define   RC_CTL_RC6_ENABLE			REG_BIT(18)
#define RC_STATE				XE_REG(0xa094)
#define RC_IDLE_HYSTERSIS			XE_REG(0xa0ac)
#define MEDIA_POWERGATE_IDLE_HYSTERESIS		XE_REG(0xa0c4)
#define RENDER_POWERGATE_IDLE_HYSTERESIS	XE_REG(0xa0c8)

#define PMINTRMSK				XE_REG(0xa168)
#define   PMINTR_DISABLE_REDIRECT_TO_GUC	REG_BIT(31)
#define   ARAT_EXPIRED_INTRMSK			REG_BIT(9)

#define FORCEWAKE_GT				XE_REG(0xa188)

#define POWERGATE_ENABLE			XE_REG(0xa210)
#define   RENDER_POWERGATE_ENABLE		REG_BIT(0)
#define   MEDIA_POWERGATE_ENABLE		REG_BIT(1)
#define   VDN_HCP_POWERGATE_ENABLE(n)		REG_BIT(3 + 2 * (n))
#define   VDN_MFXVDENC_POWERGATE_ENABLE(n)	REG_BIT(4 + 2 * (n))

#define CTC_MODE				XE_REG(0xa26c)
#define   CTC_SHIFT_PARAMETER_MASK		REG_GENMASK(2, 1)
#define   CTC_SOURCE_DIVIDE_LOGIC		REG_BIT(0)

#define FORCEWAKE_RENDER			XE_REG(0xa278)
#define FORCEWAKE_MEDIA_VDBOX(n)		XE_REG(0xa540 + (n) * 4)
#define FORCEWAKE_MEDIA_VEBOX(n)		XE_REG(0xa560 + (n) * 4)
#define FORCEWAKE_GSC				XE_REG(0xa618)

#define XEHPC_LNCFMISCCFGREG0			XE_REG_MCR(0xb01c, XE_REG_OPTION_MASKED)
#define   XEHPC_OVRLSCCC			REG_BIT(0)

/* L3 Cache Control */
#define LNCFCMOCS_REG_COUNT			32
#define XELP_LNCFCMOCS(i)			XE_REG(0xb020 + (i) * 4)
#define XEHP_LNCFCMOCS(i)			XE_REG_MCR(0xb020 + (i) * 4)
#define   L3_UPPER_LKUP_MASK			REG_BIT(23)
#define   L3_UPPER_GLBGO_MASK			REG_BIT(22)
#define   L3_UPPER_IDX_CACHEABILITY_MASK	REG_GENMASK(21, 20)
#define   L3_UPPER_IDX_SCC_MASK			REG_GENMASK(19, 17)
#define   L3_UPPER_IDX_ESC_MASK			REG_BIT(16)
#define   L3_LKUP_MASK				REG_BIT(7)
#define   L3_LKUP(value)			REG_FIELD_PREP(L3_LKUP_MASK, value)
#define   L3_GLBGO_MASK				REG_BIT(6)
#define   L3_GLBGO(value)			REG_FIELD_PREP(L3_GLBGO_MASK, value)
#define   L3_CACHEABILITY_MASK			REG_GENMASK(5, 4)
#define   L3_CACHEABILITY(value)		REG_FIELD_PREP(L3_CACHEABILITY_MASK, value)
#define   L3_SCC_MASK				REG_GENMASK(3, 1)
#define   L3_SCC(value)				REG_FIELD_PREP(L3_SCC_MASK, value)
#define   L3_ESC_MASK				REG_BIT(0)
#define   L3_ESC(value)				REG_FIELD_PREP(L3_ESC_MASK, value)

#define XEHP_L3NODEARBCFG			XE_REG_MCR(0xb0b4)
#define   XEHP_LNESPARE				REG_BIT(19)

#define L3SQCREG2				XE_REG_MCR(0xb104)
#define   COMPMEMRD256BOVRFETCHEN		REG_BIT(20)

#define L3SQCREG3				XE_REG_MCR(0xb108)
#define   COMPPWOVERFETCHEN			REG_BIT(28)

#define XEHP_L3SQCREG5				XE_REG_MCR(0xb158)
#define   L3_PWM_TIMER_INIT_VAL_MASK		REG_GENMASK(9, 0)

#define XEHP_L3SCQREG7				XE_REG_MCR(0xb188)
#define   BLEND_FILL_CACHING_OPT_DIS		REG_BIT(3)

#define XEHPC_L3CLOS_MASK(i)			XE_REG_MCR(0xb194 + (i) * 8)

#define XE2_GLOBAL_INVAL			XE_REG(0xb404)

#define SCRATCH1LPFC				XE_REG(0xb474)
#define   EN_L3_RW_CCS_CACHE_FLUSH		REG_BIT(0)

#define XE2LPM_L3SQCREG5			XE_REG_MCR(0xb658)

#define XE2_TDF_CTRL				XE_REG(0xb418)
#define   TRANSIENT_FLUSH_REQUEST		REG_BIT(0)

#define XEHP_MERT_MOD_CTRL			XE_REG_MCR(0xcf28)
#define RENDER_MOD_CTRL				XE_REG_MCR(0xcf2c)
#define COMP_MOD_CTRL				XE_REG_MCR(0xcf30)
#define XEHP_VDBX_MOD_CTRL			XE_REG_MCR(0xcf34)
#define XELPMP_VDBX_MOD_CTRL			XE_REG(0xcf34)
#define XEHP_VEBX_MOD_CTRL			XE_REG_MCR(0xcf38)
#define XELPMP_VEBX_MOD_CTRL			XE_REG(0xcf38)
#define   FORCE_MISS_FTLB			REG_BIT(3)

#define XEHP_GAMSTLB_CTRL			XE_REG_MCR(0xcf4c)
#define   CONTROL_BLOCK_CLKGATE_DIS		REG_BIT(12)
#define   EGRESS_BLOCK_CLKGATE_DIS		REG_BIT(11)
#define   TAG_BLOCK_CLKGATE_DIS			REG_BIT(7)

#define XEHP_GAMCNTRL_CTRL			XE_REG_MCR(0xcf54)
#define   INVALIDATION_BROADCAST_MODE_DIS	REG_BIT(12)
#define   GLOBAL_INVALIDATION_MODE		REG_BIT(2)

#define LMEM_CFG				XE_REG(0xcf58)
#define   LMEM_EN				REG_BIT(31)
#define   LMTT_DIR_PTR				REG_GENMASK(30, 0) /* in multiples of 64KB */

#define HALF_SLICE_CHICKEN5			XE_REG_MCR(0xe188, XE_REG_OPTION_MASKED)
#define   DISABLE_SAMPLE_G_PERFORMANCE		REG_BIT(0)

#define SAMPLER_INSTDONE			XE_REG_MCR(0xe160)
#define ROW_INSTDONE				XE_REG_MCR(0xe164)

#define SAMPLER_MODE				XE_REG_MCR(0xe18c, XE_REG_OPTION_MASKED)
#define   ENABLE_SMALLPL			REG_BIT(15)
#define   SC_DISABLE_POWER_OPTIMIZATION_EBB	REG_BIT(9)
#define   SAMPLER_ENABLE_HEADLESS_MSG		REG_BIT(5)
#define   INDIRECT_STATE_BASE_ADDR_OVERRIDE	REG_BIT(0)

#define HALF_SLICE_CHICKEN7				XE_REG_MCR(0xe194, XE_REG_OPTION_MASKED)
#define   DG2_DISABLE_ROUND_ENABLE_ALLOW_FOR_SSLA	REG_BIT(15)
#define   CLEAR_OPTIMIZATION_DISABLE			REG_BIT(6)

#define CACHE_MODE_SS				XE_REG_MCR(0xe420, XE_REG_OPTION_MASKED)
#define   DISABLE_ECC				REG_BIT(5)
#define   ENABLE_PREFETCH_INTO_IC		REG_BIT(3)

#define ROW_CHICKEN4				XE_REG_MCR(0xe48c, XE_REG_OPTION_MASKED)
#define   DISABLE_GRF_CLEAR			REG_BIT(13)
#define   XEHP_DIS_BBL_SYSPIPE			REG_BIT(11)
#define   DISABLE_TDL_PUSH			REG_BIT(9)
#define   DIS_PICK_2ND_EU			REG_BIT(7)
#define   DISABLE_HDR_PAST_PAYLOAD_HOLD_FIX	REG_BIT(4)
#define   THREAD_EX_ARB_MODE			REG_GENMASK(3, 2)
#define   THREAD_EX_ARB_MODE_RR_AFTER_DEP	REG_FIELD_PREP(THREAD_EX_ARB_MODE, 0x2)

#define ROW_CHICKEN3				XE_REG_MCR(0xe49c, XE_REG_OPTION_MASKED)
#define   XE2_EUPEND_CHK_FLUSH_DIS		REG_BIT(14)
#define   DIS_FIX_EOT1_FLUSH			REG_BIT(9)

#define TDL_TSL_CHICKEN				XE_REG_MCR(0xe4c4, XE_REG_OPTION_MASKED)
#define   STK_ID_RESTRICT			REG_BIT(12)
#define   SLM_WMTP_RESTORE			REG_BIT(11)

#define ROW_CHICKEN				XE_REG_MCR(0xe4f0, XE_REG_OPTION_MASKED)
#define   UGM_BACKUP_MODE			REG_BIT(13)
#define   MDQ_ARBITRATION_MODE			REG_BIT(12)
#define   STALL_DOP_GATING_DISABLE		REG_BIT(5)
#define   EARLY_EOT_DIS				REG_BIT(1)

#define ROW_CHICKEN2				XE_REG_MCR(0xe4f4, XE_REG_OPTION_MASKED)
#define   DISABLE_READ_SUPPRESSION		REG_BIT(15)
#define   DISABLE_EARLY_READ			REG_BIT(14)
#define   ENABLE_LARGE_GRF_MODE			REG_BIT(12)
#define   PUSH_CONST_DEREF_HOLD_DIS		REG_BIT(8)
#define   DISABLE_TDL_SVHS_GATING		REG_BIT(1)
#define   DISABLE_DOP_GATING			REG_BIT(0)

#define RT_CTRL					XE_REG_MCR(0xe530)
#define   DIS_NULL_QUERY			REG_BIT(10)

#define EU_SYSTOLIC_LIC_THROTTLE_CTL_WITH_LOCK	XE_REG_MCR(0xe534)
#define   EU_SYSTOLIC_LIC_THROTTLE_CTL_LOCK_BIT	REG_BIT(31)

#define XEHP_HDC_CHICKEN0					XE_REG_MCR(0xe5f0, XE_REG_OPTION_MASKED)
#define   LSC_L1_FLUSH_CTL_3D_DATAPORT_FLUSH_EVENTS_MASK	REG_GENMASK(13, 11)
#define   DIS_ATOMIC_CHAINING_TYPED_WRITES	REG_BIT(3)

#define LSC_CHICKEN_BIT_0			XE_REG_MCR(0xe7c8)
#define   DISABLE_D8_D16_COASLESCE		REG_BIT(30)
#define   WR_REQ_CHAINING_DIS			REG_BIT(26)
#define   TGM_WRITE_EOM_FORCE			REG_BIT(17)
#define   FORCE_1_SUB_MESSAGE_PER_FRAGMENT	REG_BIT(15)
#define   SEQUENTIAL_ACCESS_UPGRADE_DISABLE	REG_BIT(13)

#define LSC_CHICKEN_BIT_0_UDW			XE_REG_MCR(0xe7c8 + 4)
#define   UGM_FRAGMENT_THRESHOLD_TO_3		REG_BIT(58 - 32)
#define   DIS_CHAIN_2XSIMD8			REG_BIT(55 - 32)
#define   XE2_ALLOC_DPA_STARVE_FIX_DIS		REG_BIT(47 - 32)
#define   ENABLE_SMP_LD_RENDER_SURFACE_CONTROL	REG_BIT(44 - 32)
#define   FORCE_SLM_FENCE_SCOPE_TO_TILE		REG_BIT(42 - 32)
#define   FORCE_UGM_FENCE_SCOPE_TO_TILE		REG_BIT(41 - 32)
#define   MAXREQS_PER_BANK			REG_GENMASK(39 - 32, 37 - 32)
#define   DISABLE_128B_EVICTION_COMMAND_UDW	REG_BIT(36 - 32)

#define SARB_CHICKEN1				XE_REG_MCR(0xe90c)
#define   COMP_CKN_IN				REG_GENMASK(30, 29)

#define RCU_MODE				XE_REG(0x14800, XE_REG_OPTION_MASKED)
#define   RCU_MODE_FIXED_SLICE_CCS_MODE		REG_BIT(1)
#define   RCU_MODE_CCS_ENABLE			REG_BIT(0)

/*
 * Total of 4 cslices, where each cslice is in the form:
 *   [0-3]     CCS ID
 *   [4-6]     RSVD
 *   [7]       Disabled
 */
#define CCS_MODE				XE_REG(0x14804)
#define   CCS_MODE_CSLICE_0_3_MASK		REG_GENMASK(11, 0) /* 3 bits per cslice */
#define   CCS_MODE_CSLICE_MASK			0x7 /* CCS0-3 + rsvd */
#define   CCS_MODE_CSLICE_WIDTH			ilog2(CCS_MODE_CSLICE_MASK + 1)
#define   CCS_MODE_CSLICE(cslice, ccs) \
	((ccs) << ((cslice) * CCS_MODE_CSLICE_WIDTH))

#define FORCEWAKE_ACK_GT			XE_REG(0x130044)

/* Applicable for all FORCEWAKE_DOMAIN and FORCEWAKE_ACK_DOMAIN regs */
#define   FORCEWAKE_KERNEL			0
#define   FORCEWAKE_MT(bit)			BIT(bit)
#define   FORCEWAKE_MT_MASK(bit)		BIT((bit) + 16)

#define MTL_MEDIA_PERF_LIMIT_REASONS		XE_REG(0x138030)
#define MTL_MEDIA_MC6				XE_REG(0x138048)

#define GT_CORE_STATUS				XE_REG(0x138060)
#define   RCN_MASK				REG_GENMASK(2, 0)
#define   GT_C0					0
#define   GT_C6					3

#define GT_GFX_RC6_LOCKED			XE_REG(0x138104)
#define GT_GFX_RC6				XE_REG(0x138108)

#define GT0_PERF_LIMIT_REASONS			XE_REG(0x1381a8)
#define   GT0_PERF_LIMIT_REASONS_MASK		0xde3
#define   PROCHOT_MASK				REG_BIT(0)
#define   THERMAL_LIMIT_MASK			REG_BIT(1)
#define   RATL_MASK				REG_BIT(5)
#define   VR_THERMALERT_MASK			REG_BIT(6)
#define   VR_TDC_MASK				REG_BIT(7)
#define   POWER_LIMIT_4_MASK			REG_BIT(8)
#define   POWER_LIMIT_1_MASK			REG_BIT(10)
#define   POWER_LIMIT_2_MASK			REG_BIT(11)

#define GT_PERF_STATUS				XE_REG(0x1381b4)
#define   VOLTAGE_MASK				REG_GENMASK(10, 0)

/*
 * Note: Interrupt registers 1900xx are VF accessible only until version 12.50.
 *       On newer platforms, VFs are using memory-based interrupts instead.
 *       However, for simplicity we keep this XE_REG_OPTION_VF tag intact.
 */

#define GT_INTR_DW(x)				XE_REG(0x190018 + ((x) * 4), XE_REG_OPTION_VF)
#define   INTR_GSC				REG_BIT(31)
#define   INTR_GUC				REG_BIT(25)
#define   INTR_MGUC				REG_BIT(24)
#define   INTR_BCS8				REG_BIT(23)
#define   INTR_BCS(x)				REG_BIT(15 - (x))
#define   INTR_CCS(x)				REG_BIT(4 + (x))
#define   INTR_RCS0				REG_BIT(0)
#define   INTR_VECS(x)				REG_BIT(31 - (x))
#define   INTR_VCS(x)				REG_BIT(x)

#define RENDER_COPY_INTR_ENABLE			XE_REG(0x190030, XE_REG_OPTION_VF)
#define VCS_VECS_INTR_ENABLE			XE_REG(0x190034, XE_REG_OPTION_VF)
#define GUC_SG_INTR_ENABLE			XE_REG(0x190038, XE_REG_OPTION_VF)
#define   ENGINE1_MASK				REG_GENMASK(31, 16)
#define   ENGINE0_MASK				REG_GENMASK(15, 0)
#define GPM_WGBOXPERF_INTR_ENABLE		XE_REG(0x19003c, XE_REG_OPTION_VF)
#define GUNIT_GSC_INTR_ENABLE			XE_REG(0x190044, XE_REG_OPTION_VF)
#define CCS_RSVD_INTR_ENABLE			XE_REG(0x190048, XE_REG_OPTION_VF)

#define INTR_IDENTITY_REG(x)			XE_REG(0x190060 + ((x) * 4), XE_REG_OPTION_VF)
#define   INTR_DATA_VALID			REG_BIT(31)
#define   INTR_ENGINE_INSTANCE(x)		REG_FIELD_GET(GENMASK(25, 20), x)
#define   INTR_ENGINE_CLASS(x)			REG_FIELD_GET(GENMASK(18, 16), x)
#define   INTR_ENGINE_INTR(x)			REG_FIELD_GET(GENMASK(15, 0), x)
#define   OTHER_GUC_INSTANCE			0
#define   OTHER_GSC_HECI2_INSTANCE		3
#define   OTHER_GSC_INSTANCE			6

#define IIR_REG_SELECTOR(x)			XE_REG(0x190070 + ((x) * 4), XE_REG_OPTION_VF)
#define RCS0_RSVD_INTR_MASK			XE_REG(0x190090, XE_REG_OPTION_VF)
#define BCS_RSVD_INTR_MASK			XE_REG(0x1900a0, XE_REG_OPTION_VF)
#define VCS0_VCS1_INTR_MASK			XE_REG(0x1900a8, XE_REG_OPTION_VF)
#define VCS2_VCS3_INTR_MASK			XE_REG(0x1900ac, XE_REG_OPTION_VF)
#define VECS0_VECS1_INTR_MASK			XE_REG(0x1900d0, XE_REG_OPTION_VF)
#define HECI2_RSVD_INTR_MASK			XE_REG(0x1900e4)
#define GUC_SG_INTR_MASK			XE_REG(0x1900e8, XE_REG_OPTION_VF)
#define GPM_WGBOXPERF_INTR_MASK			XE_REG(0x1900ec, XE_REG_OPTION_VF)
#define GUNIT_GSC_INTR_MASK			XE_REG(0x1900f4, XE_REG_OPTION_VF)
#define CCS0_CCS1_INTR_MASK			XE_REG(0x190100)
#define CCS2_CCS3_INTR_MASK			XE_REG(0x190104)
#define XEHPC_BCS1_BCS2_INTR_MASK		XE_REG(0x190110)
#define XEHPC_BCS3_BCS4_INTR_MASK		XE_REG(0x190114)
#define XEHPC_BCS5_BCS6_INTR_MASK		XE_REG(0x190118)
#define XEHPC_BCS7_BCS8_INTR_MASK		XE_REG(0x19011c)
#define   GT_WAIT_SEMAPHORE_INTERRUPT		REG_BIT(11)
#define   GT_CONTEXT_SWITCH_INTERRUPT		REG_BIT(8)
#define   GSC_ER_COMPLETE			REG_BIT(5)
#define   GT_RENDER_PIPECTL_NOTIFY_INTERRUPT	REG_BIT(4)
#define   GT_CS_MASTER_ERROR_INTERRUPT		REG_BIT(3)
#define   GT_RENDER_USER_INTERRUPT		REG_BIT(0)

#endif
